/*
* Copyright (C) 2017
* Computer Graphics Group, The Institute of Computer Graphics and Algorithms, TU Wien
* Written by Tobias Klein <tklein@cg.tuwien.ac.at>
* All rights reserved.
*/

#include "GLWidget.h"

#include "glsw.h"
#include <glm/gtc/type_ptr.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <QMouseEvent>
#include <QDir>
#include <qopenglwidget.h>
#include <qmessagebox.h>
#include <qopenglframebufferobject.h>
#include <QOpenGLTexture>
#include "MainWindow.h"

#define GL_GPU_MEM_INFO_TOTAL_AVAILABLE_MEM_NVX 0x9048
#define GL_GPU_MEM_INFO_CURRENT_AVAILABLE_MEM_NVX 0x9049

GLfloat cube_vertices[] = {
	// front
	-1.0, -1.0,  1.0,
	1.0, -1.0,  1.0,
	1.0,  1.0,  1.0,
	-1.0,  1.0,  1.0,
	// back
	-1.0, -1.0, -1.0,
	1.0, -1.0, -1.0,
	1.0,  1.0, -1.0,
	-1.0,  1.0, -1.0,
};

GLfloat tex_coordinates[] = {
	// front
	0.0, 0.0,  1.0,
	1.0, 0.0,  1.0,
	1.0,  1.0,  1.0,
	0.0,  1.0,  1.0,
	// back
	0.0, 0.0, 0.0,
	1.0, 0.0, 0.0,
	1.0,  1.0, 0.0,
	0.0,  1.0, 0.0,
};

GLushort cube_elements[] = {
	// front
	0, 1, 2,
	2, 3, 0,
	// top
	1, 5, 6,
	6, 2, 1,
	// back
	7, 6, 5,
	5, 4, 7,
	// bottom
	4, 0, 3,
	3, 7, 4,
	// left
	4, 5, 1,
	1, 0, 4,
	// right
	3, 2, 6,
	6, 7, 3,
};

GLfloat quad_vertices[] =
{
	-1.0f, -1.0f, 0.0f,
	1.0f, -1.0f, 0.0f,
	1.0f, 1.0f, 0.0f,
	-1.0f, 1.0f, 0.0f
};

GLushort quad_elements[] =
{
	0, 1, 2, 
	0, 2, 3
};


const QString shaderPath1 = "/src/shader/cube.glsl";
const QString shaderPath2 = "/src/shader/raycasting.glsl";
const QString shaderPath3 = "/src/shader/PostProcessShader.glsl";
const QString shaderPath4 = "/src/shader/PreintegrationShader.glsl";
const QString shaderPath5 = "/src/shader/RayPropagationShader.glsl";
const QString shaderPath6 = "/src/shader/SetupShader.glsl";
const QString shaderPath7 = "/src/shader/skyboxShader.glsl";

#define WIDTH 512
#define HEIGHT 512
//#define WIDTH 1024
//#define HEIGHT 1024



GLWidget::GLWidget(QWidget *parent, MainWindow *mainWindow)
	: QOpenGLWidget(parent),
	m_vertexBuffer(QOpenGLBuffer::VertexBuffer),
	m_indexBuffer(QOpenGLBuffer::IndexBuffer),
	m_vertexBufferQuad(QOpenGLBuffer::VertexBuffer),
	m_indexBufferQuad(QOpenGLBuffer::IndexBuffer)
{
	m_MainWindow = mainWindow;
	m_renderingMode = 0;
	m_VolumeTexture = NULL;
	QSurfaceFormat format;
	// asks for a OpenGL 3.2 debug context using the Core profile
	format.setMajorVersion(4);
	format.setMinorVersion(5);
	format.setProfile(QSurfaceFormat::CoreProfile);
	format.setOption(QSurfaceFormat::DebugContext);

	setFormat(format);
	
	// we use a file watcher to recompile shaders on the fly
	// you can change shaders and directly see the result while the program is running
	m_fileWatcher = new QFileSystemWatcher(this);
	connect(m_fileWatcher, SIGNAL(fileChanged(const QString &)), this, SLOT(fileChanged(const QString &)));
	m_fileWatcher->addPath(QDir::currentPath() + shaderPath1);
	m_fileWatcher->addPath(QDir::currentPath() + shaderPath2);

	/* own */
	m_fileWatcher->addPath(QDir::currentPath() + shaderPath3);
	m_fileWatcher->addPath(QDir::currentPath() + shaderPath4);
	m_fileWatcher->addPath(QDir::currentPath() + shaderPath5);
	m_fileWatcher->addPath(QDir::currentPath() + shaderPath6);
	m_fileWatcher->addPath(QDir::currentPath() + shaderPath7);

	initglsw(); // initializes the OpenGL Shader Wrangler
	
}


GLWidget::~GLWidget()
{
	glswShutdown(); // shuts down the OpenGL Shader Wrangler
}

// initializes the OpenGL Shader Wrangler
void GLWidget::initglsw()
{
	glswInit();
	QString shader_folder = QDir::currentPath() + "/src/shader/";
	QByteArray ba = shader_folder.toLatin1();
	const char *c_str = ba.data();
	glswSetPath(c_str, ".glsl");
	qDebug() << "glsw path: " << c_str;
	glswAddDirectiveToken("", "#version 450");
}

// the rendering routing, this function is called for every frame
void GLWidget::paintGL()
{
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);

	// the matrix must be transposed when it is passed from glm to QT!
	QMatrix4x4 modelView = QMatrix4x4(glm::value_ptr(m_camera.getViewMatrix())).transposed();
	QMatrix4x4 projection = QMatrix4x4(glm::value_ptr(m_camera.getProjectionMatrix())).transposed();

	QOpenGLVertexArrayObject::Binder vaoBinder(&m_vaoCube);
	m_programCube->bind();

	m_programCube->setUniformValue(m_programCube->uniformLocation("mvMatrix"), modelView);
	m_programCube->setUniformValue(m_programCube->uniformLocation("projMatrix"), projection);

	// 1. render front faces to FBO
	//vaoBinder.rebind();
	glCullFace(GL_BACK);
	m_FBO_frontFaces->bind();
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, m_FBO_frontFaces->texture());

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glDrawElements(GL_TRIANGLES, 36, GL_UNSIGNED_SHORT, 0);

	//save fbo as image for debugging
	m_FBO_frontFaces->release();


	// 2. render back faces to FBO
	glCullFace(GL_FRONT);

	m_FBO_backFaces->bind();
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, m_FBO_backFaces->texture());

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glDrawElements(GL_TRIANGLES, 36, GL_UNSIGNED_SHORT, 0);

	//save fbo as image for debugging
	m_FBO_backFaces->release();

	//set culling back to default
	glCullFace(GL_BACK);
	//release the cube shader program
	m_vaoCube.release();
	m_programCube->release();
	vaoBinder.release();


	bool refractiveRendering = m_renderingMode == 4;
	/****************************VOLUME********************************/
	if (!refractiveRendering) {
		// 3. render the volume
		QOpenGLVertexArrayObject::Binder vaoBinderQuad(&m_vaoQuad);
		//vaoBinder2.rebind();
		m_programVolume->bind();

		//bind textures
		glActiveTexture(GL_TEXTURE0);
		glBindTexture(GL_TEXTURE_2D, m_FBO_frontFaces->texture());

		glActiveTexture(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D, m_FBO_backFaces->texture());

		glActiveTexture(GL_TEXTURE2);
		glBindTexture(GL_TEXTURE_3D, m_VolumeTexture->textureId());

		//set uniforms
		m_programVolume->setUniformValue("frontFaces", 0);
		m_programVolume->setUniformValue("backFaces", 1);
		m_programVolume->setUniformValue("volume", 2);
		m_programVolume->setUniformValue("renderingMode", m_renderingMode);

		/*******RENDERING********/
		//m_vaoQuad.bind();
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glDrawElements(GL_TRIANGLES, 36, GL_UNSIGNED_SHORT, 0);

		vaoBinderQuad.release();
		m_programVolume->release();
	}
	else {

		/****************************REFRACTIVE VOLUME VIS2********************************/

		// retrieve camera view matrix for setup shader
		QMatrix4x4 viewMatrix = QMatrix4x4(glm::value_ptr(m_camera.getViewMatrix())).transposed();
		QMatrix4x4 inverseViewMatrix = viewMatrix.inverted();
		//QMatrix4x4 viewMatrix = QMatrix4x4(glm::value_ptr(m_camera.getViewMatrix())).transposed();
		//QMatrix4x4 viewProjMatrix = QMatrix4x4(glm::value_ptr(m_camera.getProjectionMatrix() * m_camera.getViewMatrix())).inverted();
		QMatrix4x4 viewProjMatrix = QMatrix4x4(glm::value_ptr(m_camera.getProjectionMatrix() * m_camera.getViewMatrix())).transposed();
		QMatrix4x4 inverseViewProjMatrix = viewProjMatrix.transposed().inverted();
		QMatrix4x4 projMatrix = QMatrix4x4(glm::value_ptr(m_camera.getProjectionMatrix())).transposed();

		// convert glm:vec3 to QVector3D
		glm::vec3 viewPos = m_camera.getPosition();
		//std::cout << viewPos.x << ", " << viewPos.y << ", " << viewPos.z << std::endl;
		QVector3D viewPosQt = QVector3D(viewPos.x, viewPos.y, viewPos.z);
		glm::vec3 viewDir = glm::normalize(m_camera.getTarget() - viewPos);
		QVector3D viewDirQt = QVector3D(viewDir.x, viewDir.y, viewDir.z);
		float farPlaneDistance = m_camera.getFarPlane();

		m_setupBuffersProgram->bind();
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("viewMatrix"), viewMatrix);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("inverseViewMatrix"), inverseViewMatrix);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("viewProjMatrix"), viewProjMatrix);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("inverseViewProjMatrix"), inverseViewProjMatrix);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("viewPos"), viewPosQt);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("viewDir"), viewDirQt);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("farPlane"), farPlaneDistance);

		QVector3D origin = QVector3D(viewMatrix * QVector4D(0, 0, 0, 1)); // viewspace
		QVector3D startingPlanePos = origin + QVector3D(0, 0, 1); // starting point of first plane in viewspace
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("firstPlanePos"), startingPlanePos);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("origin"), origin);
		//qDebug() << origin;


		QVector3D lightColQt = QVector3D(m_lightcolor_r, m_lightcolor_g, m_lightcolor_b);
		glm::vec3 lightDir = glm::vec3(m_lightdir_x, m_lightdir_y, m_lightdir_z);
		lightDir = glm::normalize(lightDir);
		
		QVector3D lightDirQt = QVector3D(lightDir.x, lightDir.y, lightDir.z);

		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("lightCol"), lightColQt);
		m_setupBuffersProgram->setUniformValue(m_setupBuffersProgram->uniformLocation("lightDir"), lightDirQt);

		// stuff for ray calculation
		m_setupBuffersProgram->setUniformValue("fov", m_camera.getFieldOfView());
		//m_setupBuffersProgram->setUniformValue("imageWidth", width());
		//m_setupBuffersProgram->setUniformValue("imageHeight", height());
		m_setupBuffersProgram->setUniformValue("imageWidth", WIDTH);
		m_setupBuffersProgram->setUniformValue("imageHeight", HEIGHT);
		m_setupBuffersProgram->setUniformValue("aspectRatio", WIDTH / (float)HEIGHT);
		float camera_zoom = m_camera.mRadius / 30.0f;
		m_setupBuffersProgram->setUniformValue("camera_zoom", camera_zoom);

		m_refrVolRenderer->setSetupBufferUniforms(m_setupBuffersProgram, 1);

		//qDebug() << lightDirQt;

		// execute compute shader in VolumeRenderer class, because the inherited class GLWidget does not know glDispatchCompute()
		VolumeRenderer::executeComputeShader(WIDTH / 16, HEIGHT / 16, 2);
		//glMemoryBarrier(GL_TEXTURE_FETCH_BARRIER_BIT);
		glMemoryBarrier(GL_SHADER_IMAGE_ACCESS_BARRIER_BIT);


		glBindFramebuffer(GL_FRAMEBUFFER, 0);
		// release init shader
		m_setupBuffersProgram->release();



		QOpenGLVertexArrayObject::Binder vaoBinderQuadRefractive(&m_vaoLightPlaneVertices);


		m_refractiveVolumeProgram->bind();

		//bind textures

		// set uniforms for all texture buffers
		m_refrVolRenderer->setBufferUniforms(m_refractiveVolumeProgram); // textures 3 - 15

		//set uniforms
		glActiveTexture(GL_TEXTURE0);
		glBindTexture(GL_TEXTURE_3D, m_VolumeTexture->textureId());
		float sample_distance = 2.0f * (1.0f / (float(m_lightPlaneNumbers) - 1));
		m_refractiveVolumeProgram->setUniformValue("volume", 0);
		m_refractiveVolumeProgram->setUniformValue("sample_distance", sample_distance); // sample distance x 2 because of how light planes are created in z axis

		m_refractiveVolumeProgram->setUniformValue("viewMatrix", viewMatrix);
		m_refractiveVolumeProgram->setUniformValue("inverseViewMatrix", inverseViewMatrix);
		m_refractiveVolumeProgram->setUniformValue("viewProjMatrix", viewProjMatrix);
		m_refractiveVolumeProgram->setUniformValue("cameraPos", viewPosQt);
		m_refractiveVolumeProgram->setUniformValue("projMatrix", projMatrix);

		m_refractiveVolumeProgram->setUniformValue("scatteringEnabled", m_scatteringEnabled);
		m_refractiveVolumeProgram->setUniformValue("filteringEnabled", m_filteringEnabled);
		m_refractiveVolumeProgram->setUniformValue("intensityCorrectionEnabled", m_correctionEnabled);
		m_refractiveVolumeProgram->setUniformValue("refractionEnabled", m_refractionEnabled);
		m_refractiveVolumeProgram->setUniformValue("camera_zoom", camera_zoom);
		/*******RENDERING********/


		//Render Textures to smaller Viewport
		glViewport(0, 0, WIDTH, HEIGHT);
		glBindFramebuffer(GL_FRAMEBUFFER, m_refrVolRenderer->m_fbo);
		//glDisable(GL_DEPTH_TEST);
		glDisable(GL_CULL_FACE);
		//glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		GLenum drawBuffers[]{ GL_COLOR_ATTACHMENT0, GL_COLOR_ATTACHMENT1, GL_COLOR_ATTACHMENT2, GL_COLOR_ATTACHMENT3, GL_COLOR_ATTACHMENT4, GL_COLOR_ATTACHMENT5, GL_COLOR_ATTACHMENT6};
		glDrawBuffers(7, drawBuffers);

		for (int i = 0; i < m_lightPlaneNumbers; i++)
		{
			//TODO set uniform planeID. Maybe faster as buffer?
			m_refractiveVolumeProgram->setUniformValue("planeID", i);
			m_refractiveVolumeProgram->setUniformValue("planeZ", startingPlanePos.z() - i * sample_distance);
			glClear(GL_DEPTH_BUFFER_BIT);
			glTextureBarrier();
			glDrawArrays(GL_POINTS, i, 1);
		}

		glBindFramebuffer(GL_FRAMEBUFFER, 0);

		vaoBinderQuadRefractive.release();
		m_refractiveVolumeProgram->release();


		/******POST PROCESSING SHADER*****/
		glViewport(0, 0, width(), height());


		QOpenGLVertexArrayObject::Binder vaoBinderPost(&m_vaoQuad);
		m_postProcessingShaderProgram->bind();

		//bind textures
		glActiveTexture(GL_TEXTURE0);
		glBindTexture(GL_TEXTURE_2D_ARRAY, m_refrVolRenderer->m_colorTexture);
		glActiveTexture(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D_ARRAY, m_refrVolRenderer->m_viewDirectionTexture);
		glActiveTexture(GL_TEXTURE2);
		glBindTexture(GL_TEXTURE_CUBE_MAP, m_refrVolRenderer->cubemapTexture);
		glActiveTexture(GL_TEXTURE3);
		glBindTexture(GL_TEXTURE_2D, m_refrVolRenderer->m_skyboxTexture);
		glActiveTexture(GL_TEXTURE4);
		glBindTexture(GL_TEXTURE_2D, m_refrVolRenderer->m_mediumColorTexture);
		m_postProcessingShaderProgram->setUniformValue("color", 0);
		m_postProcessingShaderProgram->setUniformValue("viewDirTex", 1);
		m_postProcessingShaderProgram->setUniformValue("viewProjMatrix", viewProjMatrix);
		m_postProcessingShaderProgram->setUniformValue("projMatrix", projMatrix);
		m_postProcessingShaderProgram->setUniformValue("viewMatrix", viewMatrix);
		m_postProcessingShaderProgram->setUniformValue("firstPlanePos", startingPlanePos);
		m_postProcessingShaderProgram->setUniformValue("origin", origin);
		m_postProcessingShaderProgram->setUniformValue("skybox", 2);
		m_postProcessingShaderProgram->setUniformValue("skybox_texture", 3);
		m_postProcessingShaderProgram->setUniformValue("mediumTexture", 4);

		//glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, 0);

		vaoBinderPost.release();
		m_postProcessingShaderProgram->release();


		/* SKYBOX */
		/*
		glDepthFunc(GL_LEQUAL);
		m_skyboxProgram->bind();

		//glBindFramebuffer(GL_FRAMEBUFFER, m_refrVolRenderer->m_fbo_skybox);
		//GLenum skyboxDrawBuffer[]{ GL_COLOR_ATTACHMENT0 };
		//glDrawBuffers(1, skyboxDrawBuffer);

		//glActiveTexture(GL_TEXTURE0);
		//glBindTexture(GL_TEXTURE_2D, m_refrVolRenderer->m_skyboxTexture);
		//m_refractiveVolumeProgram->setUniformValue("skybox_buffer", 0);

		glBindVertexArray(m_refrVolRenderer->skyboxVAO);
		glActiveTexture(GL_TEXTURE0);
		glBindTexture(GL_TEXTURE_CUBE_MAP, m_refrVolRenderer->cubemapTexture);
		m_skyboxProgram->setUniformValue("skybox", 0);
		m_skyboxProgram->setUniformValue("view", viewMatrix);
		m_skyboxProgram->setUniformValue("projection", projMatrix);

		glDrawArrays(GL_TRIANGLES, 0, 36);
		glBindVertexArray(0);
		glDepthFunc(GL_LESS);

		//glBindFramebuffer(GL_FRAMEBUFFER, 0);
		m_skyboxProgram->release();
		/* SKYBOX END */

	}
}

void GLWidget::initializeGL()
{
	
	connect(context(), &QOpenGLContext::aboutToBeDestroyed, this, &GLWidget::cleanup);

	m_logger = new QOpenGLDebugLogger(this);
	m_logger->initialize();
	connect(m_logger, &QOpenGLDebugLogger::messageLogged, this, &GLWidget::handleLoggedMessage, Qt::DirectConnection);

	QWidget::setFocusPolicy(Qt::FocusPolicy::ClickFocus);

	m_VolumeTexture = new QOpenGLTexture(QOpenGLTexture::Target3D);
	m_VolumeTexture->setMinMagFilters(QOpenGLTexture::Linear, QOpenGLTexture::Linear);
	m_VolumeTexture->setFormat(QOpenGLTexture::RGB32F);
	m_VolumeTexture->setWrapMode(QOpenGLTexture::ClampToBorder);
	
	initializeOpenGLFunctions();
	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_CULL_FACE);
	glCullFace(GL_FRONT);
	glDisable(GL_BLEND);
	//glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); 


	// create programs
	m_programCube = new QOpenGLShaderProgram();
	m_programVolume = new QOpenGLShaderProgram();

	loadShaders();
	
	// create Frame Buffer Object (FBO)
	m_FBO_frontFaces = new QOpenGLFramebufferObject(width(), height());
	m_FBO_backFaces = new QOpenGLFramebufferObject(width(), height());
	
	
	if (!m_vaoCube.create()) {
		qDebug() << "error creating vao";
	}
	

	// bind vertex array object (VAO)
	// the VAO encapsulates all the data that is associated with the vertex processor
	// see: http://ogldev.atspace.co.uk/www/tutorial32/tutorial32.html
	QOpenGLVertexArrayObject::Binder vaoBinder(&m_vaoCube);
	
	// Setup our vertex buffer object
	m_vertexBuffer.create();
	m_vertexBuffer.bind();
	// Setup our index buffer object
	m_indexBuffer.create();
	m_indexBuffer.bind();

	// allocate vertices and indices of a cube
	m_vertexBuffer.allocate(cube_vertices, 8 * 3 * sizeof(GLfloat));
	m_indexBuffer.allocate(cube_elements, 12 * 3 * sizeof(GLushort)); 

	m_programCube->enableAttributeArray(0);
	m_programCube->enableAttributeArray(1);

	m_programCube->setAttributeBuffer(
		0,			// Specifies the index of the generic vertex attribute to be modified
		GL_FLOAT,	// Specifies the data type of each component in the array
		0,			// Specifies the byte offset between consecutive generic vertex attributes
		3			// Specifies the number of components per generic vertex attribute
	);


	// Setup our tex coord buffer object
	m_texCoordBuffer.create();
	m_texCoordBuffer.bind();

	m_texCoordBuffer.allocate(tex_coordinates, 8 * 3 * sizeof(GLfloat));
	
	m_programCube->setAttributeBuffer(
		1,			// Specifies the index of the generic vertex attribute to be modified
		GL_FLOAT,	// Specifies the data type of each component in the array
		0,			// Specifies the byte offset between consecutive generic vertex attributes
		3			// Specifies the number of components per generic vertex attribute
	);

	if (!m_vaoQuad.create()) {
		qDebug() << "error creating vao";
	}
	
	QOpenGLVertexArrayObject::Binder vaoBinder2(&m_vaoQuad);
	

	m_vertexBufferQuad.create();
	m_vertexBufferQuad.bind();
	
	// Setup our index buffer object
	m_indexBufferQuad.create();
	m_indexBufferQuad.bind();

	m_vertexBufferQuad.allocate(quad_vertices, 4 * 3 * sizeof(GLfloat));
	m_indexBufferQuad.allocate(quad_elements, 2 * 3 * sizeof(GLushort));

	m_programVolume->setAttributeBuffer(
		0,			// Specifies the index of the generic vertex attribute to be modified
		GL_FLOAT,	// Specifies the data type of each component in the array
		0,			// Specifies the byte offset between consecutive generic vertex attributes
		3			// Specifies the number of components per generic vertex attribute
	);

	m_programVolume->enableAttributeArray(0);
	
	GLint total_mem_kb = 0;
	glGetIntegerv(GL_GPU_MEM_INFO_TOTAL_AVAILABLE_MEM_NVX,
		&total_mem_kb);

	GLint cur_avail_mem_kb = 0;
	glGetIntegerv(GL_GPU_MEM_INFO_CURRENT_AVAILABLE_MEM_NVX,
		&cur_avail_mem_kb);

	float cur_avail_mem_mb = float(cur_avail_mem_kb) / 1024.0f;
	float total_mem_mb = float(total_mem_kb) / 1024.0f;




	/* OWN initialize refractive rendering stuff */
	initRefractiveRendering();

	//Adding labels above would cause an exception
	glObjectLabel(GL_VERTEX_ARRAY, m_vaoCube.objectId(), -1, "VAO_Cube");
	glObjectLabel(GL_VERTEX_ARRAY, m_vaoQuad.objectId(), -1, "VAO_Quad");
	glObjectLabel(GL_FRAMEBUFFER, m_FBO_frontFaces->handle(), -1, "Framebuffer_frontFaces");
	glObjectLabel(GL_FRAMEBUFFER, m_FBO_backFaces->handle(), -1, "Framebuffer_backFaces");
}


/*
* Setup the shaders and vao objects for refractive volume rendering
*/
void GLWidget::initRefractiveRendering()
{
	glewExperimental = TRUE;
	GLenum err = glewInit();
	if (err != GLEW_OK) {
		// Problem: glewInit failed, something is seriously wrong.
		std::cout << "glewInit failed: " << glewGetErrorString(err) << endl;
		exit(1);
	}

	/* Own */
	m_setupBuffersProgram = new QOpenGLShaderProgram();
	m_preintegrationProgram = new QOpenGLShaderProgram();
	m_refractiveVolumeProgram = new QOpenGLShaderProgram();
	m_postProcessingShaderProgram = new QOpenGLShaderProgram();
	m_skyboxProgram = new QOpenGLShaderProgram();

	/* Own */
	loadSetupShader();
	loadPreintShader();
	loadRefractiveVolumeShaders();
	loadPostProcessingShaders();

	float stepsize = 1.0f / float(m_lightPlaneNumbers);
	glm::vec3 lightPlaneVertices[m_lightPlaneNumbers+1];
	//glm::vec3 camToCenter = glm::vec3(0) - m_camera.getPosition(); camToCenter = glm::normalize(camToCenter);
	for (int i = 0; i <= m_lightPlaneNumbers; i++)
	{
		// create center vertices along the camera to center ray
		//lightPlaneVertices.push_back(glm::vec3(0.0f, 0.0f, (float)i));

		//float z = float(1 - (i * stepsize * 2)); from 1 to -1
		float z = float(1 - (i * stepsize)); // from 1 to 0
		// qDebug() << "Z-val of plane " << i << ": " << float(z);
		lightPlaneVertices[i] = glm::vec3(0.0f, 0.0f, z);
		/*
		if (i < m_lightPlaneNumbers / 2) {
			float z = (float)(i - (m_lightPlaneNumbers / 2.0)) / float(m_lightPlaneNumbers / 2.0);
			
		}
		if (i == m_lightPlaneNumbers / 2) {
			lightPlaneVertices[i] = glm::vec3(0.0f, 0.0f, 0.0f);
		}
		else {
			float z = (float)(i - (m_lightPlaneNumbers / 2.0)) / float(m_lightPlaneNumbers / 2.0);
			std::cout << z << endl;
			lightPlaneVertices[i] = glm::vec3(0.0f, 0.0f, z);
		}*/
	}

	//QVector3D lightPlaneVerticesQT[m_lightPlaneNumbers];
	//for (int i = 0; i < m_lightPlaneNumbers; i++)
	//{
	//	// create center vertices along the camera to center ray
	//	//lightPlaneVertices.push_back(glm::vec3(0.0f, 0.0f, (float)i));
	//	if (i < m_lightPlaneNumbers / 2)
	//		lightPlaneVerticesQT[i] = QVector3D(0.0f, 0.0f, (float)(i - (m_lightPlaneNumbers / 2.0)) / float(m_lightPlaneNumbers / 2.0));
	//	if (i == m_lightPlaneNumbers / 2)
	//		lightPlaneVerticesQT[i] = QVector3D(0.0f, 0.0f, 0.0f);
	//	else
	//		lightPlaneVerticesQT[i] = QVector3D(0.0f, 0.0f, (float)(i - (m_lightPlaneNumbers / 2.0)) / float(m_lightPlaneNumbers / 2.0));
	//}

	if (!m_vaoLightPlaneVertices.create()) {
		qDebug() << "error creating light planes vao";
	}
	glObjectLabel(GL_VERTEX_ARRAY, m_vaoLightPlaneVertices.objectId(), -1, "VAO_Lightplanes");
	QOpenGLVertexArrayObject::Binder vaoBinder3(&m_vaoLightPlaneVertices);

	m_vertexBufferLightPlanes.create();
	m_vertexBufferLightPlanes.bind();

	m_vertexBufferLightPlanes.allocate(lightPlaneVertices, m_lightPlaneNumbers * 3 * sizeof(GLfloat));

	m_refractiveVolumeProgram->setAttributeBuffer(
		0,			// Specifies the index of the generic vertex attribute to be modified
		GL_FLOAT,	// Specifies the data type of each component in the array
		0,			// Specifies the byte offset between consecutive generic vertex attributes
		3			// Specifies the number of components per generic vertex attribute
	);

	/*m_refractiveVolumeProgram->setAttributeArray(
		0,
		lightPlaneVerticesQT
	);*/

	m_refractiveVolumeProgram->enableAttributeArray(0);

	m_refrVolRenderer = new VolumeRenderer();
	// create all texture buffers
	//m_refrVolRenderer->setFramebufferSize(width(), height());
	m_refrVolRenderer->setFramebufferSize(WIDTH, HEIGHT);
	m_refrVolRenderer->setSkyboxFramebufferSize(width(), height());
	m_refrVolRenderer->InitBuffers();

	// Add a lightsource for refractive rendering (should be in the same hemisphere as the camera and parameters should be changeable -- still TODO)
	m_lightSource = new LightSource(glm::vec3(1.0, 1.0, 1.0), glm::vec3(1.0, 1.0, 0));

	m_MainWindow->m_colorTransferFunctionWidget->m_texture_handle = m_refrVolRenderer->m_preintTableColor;
	m_MainWindow->m_mediumTransferFuncWidget->m_texture_handle = m_refrVolRenderer->m_preintTableMedium;
	recalculatePreintegrationTable(m_MainWindow->m_colorTransferFunctionWidget);
	recalculatePreintegrationTable(m_MainWindow->m_mediumTransferFuncWidget);

	if (!m_vaoPostprocessingVerticesQuad.create()) {
		qDebug() << "error creating post processing quad vao";
	}
	glObjectLabel(GL_VERTEX_ARRAY, m_vaoPostprocessingVerticesQuad.objectId(), -1, "VAO_PostProcessing");
	QOpenGLVertexArrayObject::Binder vaoBinderPostQuad(&m_vaoPostprocessingVerticesQuad);

	m_vertexBufferPost.create();
	m_vertexBufferPost.bind();
	m_vertexBufferPost.allocate(quad_vertices, 4 * 3 * sizeof(GLfloat));

	// Setup our index buffer object
	m_indexBufferPost.create();
	m_indexBufferPost.bind();
	m_indexBufferPost.allocate(quad_elements, 2 * 3 * sizeof(GLushort));

	m_postProcessingShaderProgram->setAttributeBuffer(
		0,			// Specifies the index of the generic vertex attribute to be modified
		GL_FLOAT,	// Specifies the data type of each component in the array
		0,			// Specifies the byte offset between consecutive generic vertex attributes
		3			// Specifies the number of components per generic vertex attribute
	);

	m_postProcessingShaderProgram->enableAttributeArray(0);
	vaoBinderPostQuad.release();
}

void GLWidget::recalculatePreintegrationTable(TransferFunctionWidget* transferFunction) {
	auto widget = transferFunction;
	// create initial transfer function
	std::vector<glm::vec4> controlPoints;
	std::vector<glm::vec4> cpColor;

	for (int i = 0; i < 5; i++) {
		controlPoints.push_back(glm::vec4(widget->m_redValues[i].x(),widget->m_alphaValues[i].y(),1.0,1.0));
		cpColor.push_back(glm::vec4(widget->m_redValues[i].y(), widget->m_greenValues[i].y(), widget->m_blueValues[i].y(), widget->m_alphaValues[i].y()));
			
	}
	
	/*controlPoints.push_back(glm::vec4(0.0, 1.0, 1.0, 1.0));		cpColor.push_back(glm::vec3(0.0, 0.0, 1.0));
	controlPoints.push_back(glm::vec4(0.25, 1.0, 0.0, 0.4));	cpColor.push_back(glm::vec3(0.0, 1.0, 0.0));
	controlPoints.push_back(glm::vec4(0.5, 1.0, 0.0, 0.3));		cpColor.push_back(glm::vec3(0.0, 0.0, 0.0));
	controlPoints.push_back(glm::vec4(1.0, 1.0, 1.0, 1.0));		cpColor.push_back(glm::vec3(1.0, 0.9, 0.9));*/


	m_transferFunction = new TransferFunction(controlPoints, cpColor);
	m_transferFunction->setColorForControlPoints(cpColor);

	// send the texture handle for the 256x256 texture
	m_transferFunction->computePreintegrationTable(transferFunction->m_texture_handle);
}
//void GLWidget::refreshLightPlaneVertices()
//{
//	glm::vec3 lightPlaneVertices[m_lightPlaneNumbers];
//	glm::vec3 camToCenter = glm::vec3(0) - m_camera.getPosition(); camToCenter = glm::normalize(camToCenter);
//	for (int i = 0; i < m_lightPlaneNumbers; i++)
//	{
//		// create center vertices along the camera to center ray
//		//lightPlaneVertices.push_back(glm::vec3(0.0f, 0.0f, (float)i));
//		lightPlaneVertices[i] = glm::vec3(camToCenter.x, camToCenter.y, camToCenter.z) + (float)i * glm::vec3(camToCenter.x, camToCenter.y, camToCenter.z);
//	}
//
//	//if (!m_vaoLightPlaneVertices.create()) {
//	//	qDebug() << "error creating light planes vao";
//	//}
//
//	QOpenGLVertexArrayObject::Binder vaoBinder3(&m_vaoLightPlaneVertices);
//
//	//m_vertexBufferLightPlanes.create();
//	bool err = m_vertexBufferLightPlanes.bind();
//	if (!err)
//		qDebug() << "error binding vertex buffer";
//
//	m_vertexBufferLightPlanes.allocate(lightPlaneVertices, m_lightPlaneNumbers * 3 * sizeof(GLfloat));
//
//	m_refractiveVolumeProgram->setAttributeBuffer(
//		0,			// Specifies the index of the generic vertex attribute to be modified
//		GL_FLOAT,	// Specifies the data type of each component in the array
//		0,			// Specifies the byte offset between consecutive generic vertex attributes
//		3			// Specifies the number of components per generic vertex attribute
//	);
//
//	m_refractiveVolumeProgram->enableAttributeArray(0);
//
//
//}

// this function is called when the volume texture is loaded 
void GLWidget::volumeTextureLoaded(Volume *volume)
{
	makeCurrent();
	if (m_VolumeTexture != NULL) {
		delete m_VolumeTexture;
	}
	m_VolumeTexture = new QOpenGLTexture(QOpenGLTexture::Target3D);
	m_VolumeTexture->setMinMagFilters(QOpenGLTexture::Linear, QOpenGLTexture::Linear);
	m_VolumeTexture->setFormat(QOpenGLTexture::RGB32F);

	m_VolumeTexture->setSize(volume->width(), volume->height(), volume->depth());
	m_VolumeTexture->setWrapMode(QOpenGLTexture::ClampToBorder);
	m_VolumeTexture->allocateStorage(QOpenGLTexture::Red, QOpenGLTexture::Float32); // works
	m_VolumeTexture->setData(QOpenGLTexture::Red, QOpenGLTexture::Float32, volume->voxels()); // works
	doneCurrent();
}

void GLWidget::cleanup()
{
	makeCurrent();
	delete m_programCube;
	delete m_programVolume;
	delete m_refractiveVolumeProgram;
	delete m_postProcessingShaderProgram;
	delete m_skyboxProgram;
	doneCurrent();
}

glm::vec3 GLWidget::recalculateLight()
{
	return glm::vec3();
}

void GLWidget::fileChanged(const QString &path)
{
	// reboot glsw, otherwise it will use the old cached shader
	glswShutdown();
	initglsw();

	loadShaders();
	/* Own */
	loadSetupShader();
	loadRefractiveVolumeShaders();
	loadPostProcessingShaders();
	update();
}

void GLWidget::loadSetupShader()
{
	bool success = false;

	const char* cs = glswGetShader("SetupShader.Compute");
	if (!cs)
	{
		qDebug() << "Error: not able to read setup (compute) shader";
	}
	QOpenGLShader* setupShader = new QOpenGLShader(QOpenGLShader::Compute);
	success = setupShader->compileSourceCode(cs);

	m_setupBuffersProgram->removeAllShaders();
	m_setupBuffersProgram->addShader(setupShader);
	m_setupBuffersProgram->link();
	glObjectLabel(GL_SHADER, setupShader->shaderId(), -1, "Shader_setupBuffers");
	glObjectLabel(GL_PROGRAM, m_setupBuffersProgram->programId(), -1, "Program_setupBuffers");
}

void GLWidget::loadPreintShader()
{
	bool success = false;

	const char* cs = glswGetShader("PreintegrationShader.Compute");
	if (!cs)
	{
		qDebug() << "Error: not able to read preintegration (compute) shader";
	}
	QOpenGLShader* preintShader = new QOpenGLShader(QOpenGLShader::Compute);
	success = preintShader->compileSourceCode(cs);

	m_preintegrationProgram->removeAllShaders();
	m_preintegrationProgram->addShader(preintShader);
	glObjectLabel(GL_SHADER, preintShader->shaderId(), -1, "Shader_preintegration");
	glObjectLabel(GL_PROGRAM, m_preintegrationProgram->programId(), -1, "Program_preintegration");
}

void GLWidget::loadPostProcessingShaders() {
	bool success = false;

	const char* vs = glswGetShader("PostProcessShader.Vertex");
	if (!vs)
	{
		qDebug() << "Error: not able to read PostProcessingShader (vertex) shader";
	}
	QOpenGLShader* vertexShader = new QOpenGLShader(QOpenGLShader::Vertex);
	success = vertexShader->compileSourceCode(vs);


	const char* fs = glswGetShader("PostProcessShader.Fragment");
	if (!fs)
	{
		qDebug() << "Error: not able to read PostProcessingShader (fragment) shader";
	}
	QOpenGLShader* fragmentShader = new QOpenGLShader(QOpenGLShader::Fragment);
	success = fragmentShader->compileSourceCode(fs);

	m_postProcessingShaderProgram->removeAllShaders();
	m_postProcessingShaderProgram->addShader(vertexShader);
	m_postProcessingShaderProgram->addShader(fragmentShader);
	m_postProcessingShaderProgram->link();
	glObjectLabel(GL_SHADER, vertexShader->shaderId(), -1, "Shader_postprocessing_vertex");
	glObjectLabel(GL_SHADER, fragmentShader->shaderId(), -1, "Shader_postprocessing_fragment");
	glObjectLabel(GL_PROGRAM, m_postProcessingShaderProgram->programId(), -1, "Program_postprocessing");


	vs = glswGetShader("skyboxShader.Vertex");
	if (!vs)
	{
		qDebug() << "Error: not able to read skyboxShader (vertex) shader";
	}
	vertexShader = new QOpenGLShader(QOpenGLShader::Vertex);
	success = vertexShader->compileSourceCode(vs);


	fs = glswGetShader("skyboxShader.Fragment");
	if (!fs)
	{
		qDebug() << "Error: not able to read skyboxShader (fragment) shader";
	}
	fragmentShader = new QOpenGLShader(QOpenGLShader::Fragment);
	success = fragmentShader->compileSourceCode(fs);

	m_skyboxProgram->removeAllShaders();
	m_skyboxProgram->addShader(vertexShader);
	m_skyboxProgram->addShader(fragmentShader);
	m_skyboxProgram->link();
	glObjectLabel(GL_SHADER, vertexShader->shaderId(), -1, "Shader_skybox_vertex");
	glObjectLabel(GL_SHADER, fragmentShader->shaderId(), -1, "Shader_skybox_fragment");
	glObjectLabel(GL_PROGRAM, m_skyboxProgram->programId(), -1, "Program_skybox");
}

void GLWidget::loadRefractiveVolumeShaders()
{
	bool success = false;

	const char* vs = glswGetShader("RayPropagationShader.Vertex");
	if (!vs)
	{
		qDebug() << "Error: not able to read RayPropagationShader (vertex) shader";
	}
	QOpenGLShader* vertexShader = new QOpenGLShader(QOpenGLShader::Vertex);
	success = vertexShader->compileSourceCode(vs);

	const char* lightPlaneSource = glswGetShader("RayPropagationShader.Geometry");
	if (!lightPlaneSource)
	{
		qDebug() << "Error: not able to read RayPropagationShader (geometry) shader";
	}
	QOpenGLShader* lightPlaneShader = new QOpenGLShader(QOpenGLShader::Geometry);
	success = lightPlaneShader->compileSourceCode(lightPlaneSource);

	const char* fs = glswGetShader("RayPropagationShader.Fragment");
	if (!fs)
	{
		qDebug() << "Error: not able to read RayPropagationShader (fragment) shader";
	}
	QOpenGLShader* fragmentShader = new QOpenGLShader(QOpenGLShader::Fragment);
	success = fragmentShader->compileSourceCode(fs);

	m_refractiveVolumeProgram->removeAllShaders();
	m_refractiveVolumeProgram->addShader(vertexShader);
	m_refractiveVolumeProgram->addShader(lightPlaneShader);
	m_refractiveVolumeProgram->addShader(fragmentShader);

	//TODO?
	//glBindFragDataLocation(m_refractiveVolumeProgram->programId(), 0, "color_texture_write");

	m_refractiveVolumeProgram->link();


	glObjectLabel(GL_SHADER, vertexShader->shaderId(), -1, "Shader_refractive_vertex");
	glObjectLabel(GL_SHADER, lightPlaneShader->shaderId(), -1, "Shader_refractive_geometry");
	glObjectLabel(GL_SHADER, fragmentShader->shaderId(), -1, "Shader_refractive_fragment");
	glObjectLabel(GL_PROGRAM, m_refractiveVolumeProgram->programId(), -1, "Program_refractive");
}

void GLWidget::loadShaders()
{
	bool success = false;

	const char *vs = glswGetShader("cube.Vertex");
	if (!vs)
	{
		qDebug() << "Error: not able to read vertex shader";
	}
	QOpenGLShader *vertexShader = new QOpenGLShader(QOpenGLShader::Vertex);
	success = vertexShader->compileSourceCode(vs);

	const char *fs = glswGetShader("cube.Fragment");
	if (!fs)
	{
		qDebug() << "Error: not able to read fragment shader";
	}
	QOpenGLShader *fragmentShader = new QOpenGLShader(QOpenGLShader::Fragment);
	success = fragmentShader->compileSourceCode(fs);

	m_programCube->removeAllShaders();
	m_programCube->addShader(vertexShader);
	m_programCube->addShader(fragmentShader);

	const char *vsVolume = glswGetShader("raycasting.Vertex");
	if (!vsVolume)
	{
		qDebug() << "Error: not able to read vertex shader";
	}
	QOpenGLShader *vertexShaderVolume = new QOpenGLShader(QOpenGLShader::Vertex);
	success = vertexShaderVolume->compileSourceCode(vsVolume);

	const char *fsVolume = glswGetShader("raycasting.Fragment");
	if (!fsVolume)
	{
		qDebug() << "Error: not able to read fragment shader";
	}
	QOpenGLShader *fragmentShaderVolume = new QOpenGLShader(QOpenGLShader::Fragment);
	success = fragmentShaderVolume->compileSourceCode(fsVolume);
	
	m_programVolume->removeAllShaders();
	m_programVolume->addShader(vertexShaderVolume);
	m_programVolume->addShader(fragmentShaderVolume);
	m_programVolume->link();
}

// this function is called whenever the widget has been resized
void GLWidget::resizeGL(int w, int h)
{
	// when the widget is resized, the aspect ratio of the camera needs to be updated
	m_camera.setAspect(float(w) / h);
}

void GLWidget::mousePressEvent(QMouseEvent *event)
{
	m_lastPos = event->pos();
}

void GLWidget::wheelEvent(QWheelEvent *event)
{
	m_camera.zoom(event->delta() / 100);
	update();
}

void GLWidget::mouseMoveEvent(QMouseEvent *event)
{
	int dx = event->x() - m_lastPos.x();
	int dy = event->y() - m_lastPos.y();

	if (event->buttons() & Qt::LeftButton) {
		m_camera.rotateAzimuth(dx / 100.0f);
		m_camera.rotatePolar(dy / 100.0f);
	}

	if (event->buttons() & Qt::RightButton) {
		m_camera.rotateAzimuth(dx / 100.0f);
		m_camera.rotatePolar(dy / 100.0f);
	}
	m_lastPos = event->pos();
	update();
}

void GLWidget::keyPressEvent(QKeyEvent *event)
{
	switch (event->key())
	{
	case Qt::Key_Space:
	{
		break;
	}
	default:
	{
		event->ignore();
		break;
	}
	}
}

void GLWidget::keyReleaseEvent(QKeyEvent *event)
{

}

void GLWidget::handleLoggedMessage()
{
	const QList<QOpenGLDebugMessage> messages = m_logger->loggedMessages();
	for (const QOpenGLDebugMessage &message : messages)
	{
		qDebug() << message;
	}
}

