
/* Copyright (c) Nate Robins, 1997. */

/* This program is freely distributable without licensing fees
   and is provided without guarantee or warrantee expressed or
   implied. This program is -not- in the public domain. */

#include "glutint.h"

/* global current HDC */
extern HDC XHDC;

/* MW: hack: global variable which tells __glutDetermineWindowVisual
   whether we want a visual for a pbuffer */
int __pbufferactive = False;

GLXContext
glXCreateContext(Display * display, XVisualInfo * visinfo,
  GLXContext share, Bool direct)
{
  /* KLUDGE: GLX really expects a display pointer to be passed
     in as the first parameter, but Win32 needs an HDC instead,
     so BE SURE that the global XHDC is set before calling this
     routine. */
  HGLRC context;

  context = wglCreateContext(XHDC);

#if 0
  /* XXX GLUT doesn't support it now, so don't worry about display list
     and texture object sharing. */
  if (share) {
    wglShareLists(share, context);
  }
#endif

  /* Since direct rendering is implicit, the direct flag is
     ignored. */

  return context;
}

/* MW: checking for OpenGL support is more difficult now that we have
   to check for multisample. The reason is that the multisample pixelformats
   don't advertise OpenGL-capability via DescribePixelFormat, only via
   the new WGL_ARB_pixelformat mechanism */
static int __getSamples(int index);
static int __getSupportOpenGL(int index);

int
glXGetConfig(Display * display, XVisualInfo * visual, int attrib, int *value)
{
  if (!visual)
    return GLX_BAD_VISUAL;

  switch (attrib) {
  case GLX_USE_GL:
    if (    (visual->dwFlags & PFD_DRAW_TO_WINDOW)
        /* need to ask ARB_pixelformat first because of pbuffers and multisample */
        && ( __getSupportOpenGL(visual->dwDamageMask) ||
             (visual->dwFlags & PFD_SUPPORT_OPENGL) && !__pbufferactive )
       )
    {
      /* XXX Brad's Matrix Millenium II has problems creating
         color index windows in 24-bit mode (lead to GDI crash)
         and 32-bit mode (lead to black window).  The cColorBits
         filed of the PIXELFORMATDESCRIPTOR returned claims to
         have 24 and 32 bits respectively of color indices. 2^24
         and 2^32 are ridiculously huge writable colormaps.
         Assume that if we get back a color index
         PIXELFORMATDESCRIPTOR with 24 or more bits, the
         PIXELFORMATDESCRIPTOR doesn't really work and skip it.
         -mjk */
      if (visual->iPixelType == PFD_TYPE_COLORINDEX
        && visual->cColorBits >= 24) {
        *value = 0;
      } else {
    *value = 1;
      }
    } else {
      *value = 0;
    }
    break;
  case GLX_BUFFER_SIZE:
    /* KLUDGE: if we're RGBA, return the number of bits/pixel,
       otherwise, return 8 (we guessed at 256 colors in CI
       mode). */
    if (visual->iPixelType == PFD_TYPE_RGBA)
      *value = visual->cColorBits;
    else
      *value = 8;
    break;
  case GLX_LEVEL:
    /* The bReserved flag of the pfd contains the
       overlay/underlay info. */
    *value = visual->bReserved;
    break;
  case GLX_RGBA:
    *value = visual->iPixelType == PFD_TYPE_RGBA;
    break;
  case GLX_DOUBLEBUFFER:
    *value = visual->dwFlags & PFD_DOUBLEBUFFER;
    break;
  /* MW: test for MCD or ICD */
  case GLX_HARDWARE_ACCELERATED:
    *value = ((visual->dwFlags & PFD_GENERIC_ACCELERATED) || !(visual->dwFlags & PFD_GENERIC_FORMAT));
    break;
  /* MW: swap mode support */
  case GLX_SWAP_MODE:
    if (visual->dwFlags & PFD_SWAP_COPY)
        *value = 2;
    else if (visual->dwFlags & PFD_SWAP_EXCHANGE)
        *value = 1;
    else
        *value = 0;
    break;
  case GLX_STEREO:
    /* MW: this test returned "2", where it should have returned "1"!
       this breaks glutInitDisplayString because the STEREO capability is tested
       for == 1 */
    *value = (visual->dwFlags & PFD_STEREO) == PFD_STEREO;
    break;
  case GLX_AUX_BUFFERS:
    *value = visual->cAuxBuffers;
    break;
  case GLX_RED_SIZE:
    *value = visual->cRedBits;
    break;
  case GLX_GREEN_SIZE:
    *value = visual->cGreenBits;
    break;
  case GLX_BLUE_SIZE:
    *value = visual->cBlueBits;
    break;
  case GLX_ALPHA_SIZE:
    *value = visual->cAlphaBits;
    break;
  case GLX_DEPTH_SIZE:
    *value = visual->cDepthBits;
    break;
  case GLX_STENCIL_SIZE:
    *value = visual->cStencilBits;
    break;
  case GLX_ACCUM_RED_SIZE:
    *value = visual->cAccumRedBits;
    break;
  case GLX_ACCUM_GREEN_SIZE:
    *value = visual->cAccumGreenBits;
    break;
  case GLX_ACCUM_BLUE_SIZE:
    *value = visual->cAccumBlueBits;
    break;
  case GLX_ACCUM_ALPHA_SIZE:
    *value = visual->cAccumAlphaBits;
    break;
  /* MW: mulstisample support in WIN32 */
  case GLX_SAMPLES_SGIS:
    *value = __getSamples(visual->dwDamageMask);
    break;
  default:
    return GLX_BAD_ATTRIB;
  }
  return 0;
}

XVisualInfo *
glXChooseVisual(Display * display, int screen, int *attribList)
{
  /* KLUDGE: since we need the HDC, MAKE SURE to set XHDC
     before calling this routine. */

  int *p = attribList;
  int pf;
  PIXELFORMATDESCRIPTOR pfd;
  PIXELFORMATDESCRIPTOR *match = NULL;
  int stereo = 0;

  /* Avoid seg-faults. */
  if (!p)
    return NULL;

  memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
  pfd.nSize = (sizeof(PIXELFORMATDESCRIPTOR));
  pfd.nVersion = 1;

  /* Defaults. */
  pfd.dwFlags = PFD_SUPPORT_OPENGL | PFD_DRAW_TO_WINDOW;
  pfd.iPixelType = PFD_TYPE_COLORINDEX;
  pfd.cColorBits = 32;
  pfd.cDepthBits = 0;

  while (*p) {
    switch (*p) {
    case GLX_USE_GL:
      pfd.dwFlags |= PFD_SUPPORT_OPENGL;
      break;
    case GLX_BUFFER_SIZE:
      pfd.cColorBits = *(++p);
      break;
    case GLX_LEVEL:
      /* the bReserved flag of the pfd contains the
         overlay/underlay info. */
      pfd.bReserved = *(++p);
      break;
    case GLX_RGBA:
      pfd.iPixelType = PFD_TYPE_RGBA;
      break;
    case GLX_DOUBLEBUFFER:
      pfd.dwFlags |= PFD_DOUBLEBUFFER;
      break;
    case GLX_STEREO:
      stereo = 1;
      pfd.dwFlags |= PFD_STEREO;
      break;
    case GLX_AUX_BUFFERS:
      pfd.cAuxBuffers = *(++p);
      break;
    case GLX_RED_SIZE:
      pfd.cRedBits = 8; /* Try to get the maximum. */
      ++p;
      break;
    case GLX_GREEN_SIZE:
      pfd.cGreenBits = 8;
      ++p;
      break;
    case GLX_BLUE_SIZE:
      pfd.cBlueBits = 8;
      ++p;
      break;
    case GLX_ALPHA_SIZE:
      pfd.cAlphaBits = 8;
      ++p;
      break;
    case GLX_DEPTH_SIZE:
      pfd.cDepthBits = 32;
      ++p;
      break;
    case GLX_STENCIL_SIZE:
      pfd.cStencilBits = *(++p);
      break;
    case GLX_ACCUM_RED_SIZE:
    case GLX_ACCUM_GREEN_SIZE:
    case GLX_ACCUM_BLUE_SIZE:
    case GLX_ACCUM_ALPHA_SIZE:
      /* I believe that WGL only used the cAccumRedBits,
     cAccumBlueBits, cAccumGreenBits, and cAccumAlphaBits fields
     when returning info about the accumulation buffer precision.
     Only cAccumBits is used for requesting an accumulation
     buffer. */
      pfd.cAccumBits = 1;
      ++p;
      break;
    }
    ++p;
  }

  /* Let Win32 choose one for us. */
  pf = ChoosePixelFormat(XHDC, &pfd);
  if (pf > 0) {
    match = (PIXELFORMATDESCRIPTOR *) malloc(sizeof(PIXELFORMATDESCRIPTOR));
    DescribePixelFormat(XHDC, pf, sizeof(PIXELFORMATDESCRIPTOR), match);

    /* ChoosePixelFormat is dumb in that it will return a pixel
       format that doesn't have stereo even if it was requested
       so we need to make sure that if stereo was selected, we
       got it. */
    if (stereo) {
      if (!(match->dwFlags & PFD_STEREO)) {
        free(match);
    return NULL;
      }
    }
    /* XXX Brad's Matrix Millenium II has problems creating
       color index windows in 24-bit mode (lead to GDI crash)
       and 32-bit mode (lead to black window).  The cColorBits
       filed of the PIXELFORMATDESCRIPTOR returned claims to
       have 24 and 32 bits respectively of color indices. 2^24
       and 2^32 are ridiculously huge writable colormaps.
       Assume that if we get back a color index
       PIXELFORMATDESCRIPTOR with 24 or more bits, the
       PIXELFORMATDESCRIPTOR doesn't really work and skip it.
       -mjk */
    if (match->iPixelType == PFD_TYPE_COLORINDEX
      && match->cColorBits >= 24) {
      free(match);
      return NULL;
    }
  }
  /* MW: To support the glutInitDisplayString mechanism of passing
     the pixelformat, this has to be done as well in the glutInitDisplayMode-mechanism: */
  match->dwDamageMask = pf;
  return match;
}


/* MW: The following a bit hacked to support multisampling and pbuffers in WIN32 only.
   only uses ARB extensions, could use some nicer interface. */
#include <GL/wglext.h>

/* this works for both gl and wgl strings */
static int is_extension_supported(const char *extensions, const char *extension)
{
  const char *start;
  char *where, *terminator;

  if (extensions == NULL || extension == NULL)
        return 0;

    /* It takes a bit of care to be fool-proof about parsing
       the GLX extensions string.  Don't be fooled by
       sub-strings,  etc. */
    start = extensions;
    for (;;) {
      where = strstr(start, extension);
      if (!where)
        return 0;
      terminator = where + strlen(extension);
      if (where == start || *(where - 1) == ' ') {
        if (*terminator == ' ' || *terminator == '\0') {
          return 1;
        }
      }
      start = terminator;
    }

    return 0;
}


/* define the function entry points here */
PFNWGLGETPIXELFORMATATTRIBIVARBPROC wglGetPixelFormatAttribivARB = NULL;

PFNWGLCREATEPBUFFERARBPROC wglCreatePbufferARB = NULL;
PFNWGLGETPBUFFERDCARBPROC wglGetPbufferDCARB = NULL;
PFNWGLRELEASEPBUFFERDCARBPROC wglReleasePbufferDCARB = NULL;
PFNWGLDESTROYPBUFFERARBPROC wglDestroyPbufferARB = NULL;
PFNWGLQUERYPBUFFERARBPROC wglQueryPbufferARB = NULL;


/* init multisample and pbuffer extensions if possible, and
   return the extension string */
static const char *init_wgl_extensions()
{
    PFNWGLGETEXTENSIONSSTRINGARBPROC wglGetExtensionsStringARB;
    const char *extensions = NULL;
	static char *extension_buf = NULL;

    HGLRC ctx = NULL;
    HWND tmpwin = NULL;
    HDC hDC = wglGetCurrentDC();

    /* to get wgl extensions, we need to create a temporary window */
    if (!hDC)
    {
        int pf;
        PIXELFORMATDESCRIPTOR pfd;

        tmpwin = CreateWindow("GLUT", "tmp",
            WS_CLIPSIBLINGS | WS_CLIPCHILDREN,
            CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL,
            NULL, GetModuleHandle(NULL), 0);
        hDC = GetDC(tmpwin);

        memset(&pfd, 0, sizeof(pfd));
        pfd.nSize        = sizeof(pfd);
        pfd.nVersion     = 1;
        pfd.dwFlags      = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL;
        pf = ChoosePixelFormat(hDC, &pfd);
        SetPixelFormat(hDC, pf, &pfd);

        ctx = wglCreateContext(hDC);
        wglMakeCurrent(hDC, ctx);
    }

    wglGetExtensionsStringARB
    = (PFNWGLGETEXTENSIONSSTRINGARBPROC)wglGetProcAddress( "wglGetExtensionsStringARB" );

    /* now that we have a temporary window, go ahead and create those entry points! */
    if (wglGetExtensionsStringARB)
    {
        extensions = wglGetExtensionsStringARB( hDC );
		if (extensions)
		{
			if (extension_buf) free(extension_buf);
			extension_buf = (char *)malloc(strlen(extensions)+1);
			strcpy(extension_buf, extensions);
			extensions = extension_buf;
		}

        if (is_extension_supported(extensions, "WGL_ARB_pixel_format"))
            wglGetPixelFormatAttribivARB =
                (PFNWGLGETPIXELFORMATATTRIBIVARBPROC)wglGetProcAddress( "wglGetPixelFormatAttribivARB" );

        if (is_extension_supported(extensions, "WGL_ARB_pbuffer"))
        {
            wglCreatePbufferARB =
                (PFNWGLCREATEPBUFFERARBPROC)wglGetProcAddress( "wglCreatePbufferARB" );
            wglGetPbufferDCARB =
                (PFNWGLGETPBUFFERDCARBPROC)wglGetProcAddress( "wglGetPbufferDCARB" );
            wglReleasePbufferDCARB =
                (PFNWGLRELEASEPBUFFERDCARBPROC)wglGetProcAddress( "wglReleasePbufferDCARB" );
            wglDestroyPbufferARB =
                (PFNWGLDESTROYPBUFFERARBPROC)wglGetProcAddress( "wglDestroyPbufferARB" );
            wglQueryPbufferARB =
                (PFNWGLQUERYPBUFFERARBPROC)wglGetProcAddress( "wglQueryPbufferARB" );
        }
    }


    /* destroy the tmp window if we needed to create one */
    if (tmpwin)
    {
        wglMakeCurrent(NULL, NULL);
        wglDeleteContext(ctx);
        DestroyWindow(tmpwin);
    }

    return extensions;
}


/* mimics X behavior. we only get the extension string once */
int
__glutIsSupportedByGLX(char *extension)
{
  static const char *extensions = NULL;

  if (!extensions)
  {
      extensions = init_wgl_extensions();
  }

  /* Map SGI multisample to WGL multisample */
  if (strcmp(extension, "GLX_SGIS_multisample") == 0)
    return __glutIsSupportedByGLX("WGL_ARB_multisample") || __glutIsSupportedByGLX("WGL_EXT_multisample");

  return is_extension_supported(extensions, extension);
}


/* MW: get number of samples in a pixelformat */
static int __getSamples(int index)
{
#ifdef WGL_ARB_pixel_format
    if (wglGetPixelFormatAttribivARB)
    {
        int attrib = WGL_SAMPLES_EXT;
        int result;
        if (wglGetPixelFormatAttribivARB(XHDC, index, 0, 1, &attrib, &result))
            return result;
    }
#endif
    return 0;
}


/* determine OpenGL support for multisample pixelformats and pbuffer */
static int __getSupportOpenGL(int index)
{
#ifdef WGL_ARB_pixel_format
    if (wglGetPixelFormatAttribivARB)
    {
        int result;
        int attrib;
        if (__pbufferactive)
            attrib = WGL_DRAW_TO_PBUFFER_ARB;
        else
            attrib = WGL_SUPPORT_OPENGL_ARB;
        if (wglGetPixelFormatAttribivARB(XHDC, index, 0, 1, &attrib, &result))
            return result;
    }
#endif
    return 0;
}

/* low level destroy */
void XDestroyPBuffer(GLUTwindow *window)
{
  wglReleasePbufferDCARB( (HPBUFFERARB)window->win, window->hdc );
  wglDestroyPbufferARB( (HPBUFFERARB)window->win );
}

/* low level create */
void XCreatePBuffer(GLUTwindow *window, int format, int width, int height)
{
    int attriblist = 0;
    window->win = (Window)wglCreatePbufferARB( XHDC, format, width, height, &attriblist );

    if ( !window->win )
    {
        DWORD err = GetLastError();
        if ( err == ERROR_INVALID_PIXEL_FORMAT )
            __glutFatalError("pbuffer creation: wglCreatePbufferARB() failed: ERROR_INVALID_PIXEL_FORMAT" );
        else if ( err == ERROR_NO_SYSTEM_RESOURCES )
            __glutFatalError( "pbuffer creation: wglCreatePbufferARB() failed: ERROR_NO_SYSTEM_RESOURCES\n" );
        else if ( err == ERROR_INVALID_DATA )
            __glutFatalError( "pbuffer creation: wglCreatePbufferARB() failed: ERROR_INVALID_DATA\n" );
        else
            __glutFatalError( "pbuffer creation: wglCreatePbufferARB() failed: error code unknown!" );
    }


    window->hdc = wglGetPbufferDCARB( (HPBUFFERARB)window->win );
    if ( !window->hdc )
        __glutFatalError( "pbuffer creation error:  wglGetPbufferDCARB() failed" );

    // Determine the actual width and height we were able to create.
    wglQueryPbufferARB( (HPBUFFERARB)window->win, WGL_PBUFFER_WIDTH_ARB, &window->width );
    wglQueryPbufferARB( (HPBUFFERARB)window->win, WGL_PBUFFER_HEIGHT_ARB, &window->height );
}

int XPBufferPossible(int index, int width, int height)
{
    int result[3];
    int attribs[] = { WGL_MAX_PBUFFER_PIXELS_ARB, WGL_MAX_PBUFFER_WIDTH_ARB, WGL_MAX_PBUFFER_HEIGHT_ARB };
    if (wglGetPixelFormatAttribivARB(XHDC, index, 0, 3, attribs, result))
        return (width * height <= result[0] && width <= result[1] && height <= result[2]);

    return 0;
}

/* pbuffer surfaces can get lost on mode switches */
void XHandleModeSwitch(GLUTwindow *window)
{
    int lost = 0;

    wglQueryPbufferARB( (HPBUFFERARB)window->win, WGL_PBUFFER_LOST_ARB, &lost );

    if ( lost )
    {
        wglDeleteContext( window->ctx );
        XDestroyPBuffer(window);
        XCreatePBuffer(window, window->vis->dwDamageMask, window->width, window->height);
        window->renderDc = window->hdc;
        window->ctx = wglCreateContext( window->hdc );
        window->renderCtx = window->ctx;
    }
}

/* Note: TODO for pbuffers:

  - better glutCreateBuffers integration (parm to glutCreateWindow instead?)
  - better displaymodepossible integration (but how?)

  - check all calls whether they are compatible with pbuffers
  - make compatible with SGI!!!

  + PBuffers are not normal glut windows, i.e. they will not be put on the window
    worklist and no callback will be called for them. Therefore, some calls should not
    be made to pbuffers (the results would be undefined):
    * glutpostredisplay
    * glutPosition/Reshape/Pop/Push/Iconfiy/Show/Hide/Fullscreen/SetCursor/...

  + handle creation errors? (fatal errors at the moment)
  + how to check whether enough resources are available (within displaymodepossible at the moment)
  + how to check for actual pbuffers size? (is set into window at the moment, check with glutGet)
  + modeswitches: check all pbuffer windows at enter/leave gamemode
  + necessary changes:
    * introduced glutCreateBuffers in glut_win.c
    * changed glutDestroyWindow in glut_win.c
    * changed the draw to window flag checking (need pbufferactive for that)
    * pbuffer-flag in window
*/