/**
 * \file camera.h
 *
 * \brief This file contains the camera class.
 *
 * The class represents the scenes camera object.
 *
 * \author MAxx Headroom
 * \version 0.0
 * \date 12.01.2004 21:27:52
 *
 * \todo	init function to be called in gl-init
 * \todo	function which can be called in resize
 *
 **/

#ifndef __camera_h__
#define __camera_h__


#include "general.h"
#include "object.h"
#include "vector.h"


/**
 *
 * \brief Takes control of the camera.
 *
 **/
class Camera : public Object
{
	private:

		vectord pos;
		vectord center;
		vectord up;

		/**
		 * At this point the consoles upper left corner resides.
		 **/
		vectord consUL;
		/**
		 * This union vector points in the upwards direction on the
		 * console.
		 **/
		vectord consUp;
		/**
		 * This union vector points to the right on the console.
		 **/
		vectord consRight;
		/**
		 * Indicates, that the console vectors (consUL, consUp and
		 * consRight) do not correspond with the actual position of the
		 * camera.
		 **/
		bool consDirty;

		GLdouble movement;

		// Window geometry:
		int width;
		int height;

		/**
		 *
		 * \brief Does a recalculation of the consol vectors.
		 *
		 * \param 
		 *
		 **/
		void calcConsVectors();


		/**
		 * Hide default constructor.
		 **/
		Camera();


	public:

		Camera(int w, int h);
		~Camera();

		virtual void draw();

		virtual void update(int msec);

		void setPosition(double x, double y, double z)
		{
			pos.X = x;
			pos.Y = y;
			pos.Z = z;
			consDirty = true;
		}

		void setPosition(const vectord& v)
		{
			pos = v;
			consDirty = true;
		}

		/**
		 *
		 * \brief Set the look at position to \arg x, \arg y and \arg z.
		 *
		 **/
		void setLookAt(double x, double y, double z)
		{
			center.X = x;
			center.Y = y;
			center.Z = z;
			consDirty = true;
		}

		void setWindowGeo(int w, int h)
		{
			width = w;
			height = h;
		}

		// Consol functions:
		//	These functions serve to get information about the
		//	consol, i.e. the up vector and the upper left corner.

		/**
		 *
		 * \brief Provides the position of the upper left corner of the
		 * console.
		 *
		 **/
		vectord getConsUpperLeft();

		/**
		 *
		 * \brief Provides the consols up vector.
		 *
		 **/
		vectord getConsUpVect();

		/**
		 *
		 * \brief Provides the consols right vector.
		 *
		 **/
		vectord getConsRightVect();

}; /* Camera */

#endif /* __camera_h__ */
