#version 430 core

#define MAX_DLIGHTS 2
#define MAX_PLIGHTS 32
#define MAX_SLIGHTS 32

out vec4 fragColor;

in vec2 v_uv;

uniform sampler2D gPosition;
uniform sampler2D gNormal;
uniform sampler2D gAlbedoShine;
uniform sampler2D gKaSpecR;
uniform sampler2D gKdSpecG;
uniform sampler2D gKsSpecB;
uniform sampler2D gShadowDir;
uniform sampler2D gShadowSpot;

uniform float brightness;
uniform int displayTarget;
uniform vec3 camera_world;

uniform struct directional_light {
    vec3 color;
    vec3 direction;
} dirL[MAX_DLIGHTS];

uniform struct point_light {
    vec3 color;
    vec3 position;
    vec3 attenuation;
} pointL[MAX_PLIGHTS];

uniform struct spot_light {
    vec3 color;
    vec3 position;
    vec3 attenuation;
    vec3 direction;
    float innerAngle;
    float outerAngle;
} spotL[MAX_SLIGHTS];


vec3 phong(vec3 n, vec3 l, vec3 v, vec3 diffuseC, vec3 kd, vec3 specularC, vec3 ks, float shininess, bool attenuate, vec3 attenuation) {
    float d = length(l);
    l = normalize(l);
    float att = 1.0;
    if (attenuate) att = 1.0f / (attenuation.x + d * attenuation.y + d * d * attenuation.z);
    vec3 r = reflect(-l, n);
    return (kd * diffuseC * max(0, dot(n, l)) + ks * specularC * pow(max(0, dot(r, v)), shininess)) * att;
}

void main()
{
    if (displayTarget == 1) {
	// render only position texture
	fragColor = vec4(texture(gPosition, v_uv).rgb, 1.0);
	return;
    }
    if (displayTarget == 2) {
	// render only normal texture
	fragColor = vec4(texture(gNormal, v_uv).rgb, 1.0);
	return;
    }
    if (displayTarget == 3) {
	// render only albedo texture
	fragColor = vec4(texture(gAlbedoShine, v_uv).rgb, 1.0);
	// special case for car and ground since they don't have actual
	// textures and would appear with a default white texture instead
	// their color
	if (fragColor.r == 1.0)
	    fragColor = vec4(texture(gKdSpecG, v_uv).rgb, 1.0);
	return;
    }
    if (displayTarget == 4) {
    // render only shadow texture (directional light)
    fragColor = vec4(texture(gShadowDir, v_uv).rgb, 1.0);
    return;
    }
    if (displayTarget == 5) {
    // render only shadow texture (closest spot light)
    fragColor = vec4(texture(gShadowSpot, v_uv).rgb, 1.0);
    return;
    }

    vec3 n = normalize(texture(gNormal, v_uv).rgb);
    vec3 pos = texture(gPosition, v_uv).rgb;
    vec3 v = normalize(camera_world - pos);

    vec3 ambient = texture(gAlbedoShine, v_uv).rgb * 0.01f; // fixed ambient factor
    vec3 diffuse = texture(gAlbedoShine, v_uv).rgb;
    vec3 ka = texture(gKaSpecR, v_uv).rgb;
    vec3 kd = texture(gKdSpecG, v_uv).rgb;
    vec3 ks = texture(gKsSpecB, v_uv).rgb;

    //float shadowFactor = texture(gShadow, v_uv).r;

    // scale shininess back up to [0, 100] range
    float shininess = texture(gAlbedoShine, v_uv).a*100.0;

    vec3 specular = vec3(texture(gKaSpecR, v_uv).a,
			 texture(gKdSpecG, v_uv).a,
			 texture(gKsSpecB, v_uv).a);

    fragColor = vec4(ambient, 1);

    // add directional light contribution
    for (uint i = 0; i < dirL.length(); i++) {
	if (dirL[i].color == vec3(0)) continue;
    float shadowFactor = texture(gShadowDir, v_uv).r;
	fragColor.rgb += phong(n, -dirL[i].direction, v, dirL[i].color * brightness * diffuse, kd,
			       dirL[i].color * specular, ks, shininess, false, vec3(0)) * shadowFactor;
    }

    // add point light contribution
    for (uint i = 0; i < pointL.length(); i++) {
	if (pointL[i].color == vec3(0)) continue;
	fragColor.rgb += phong(n, (pointL[i].position - pos), v, pointL[i].color * diffuse, kd,
			       pointL[i].color * specular, ks, shininess, true, pointL[i].attenuation);
    }

    // add spot light contribution
    for (uint i = 0; i < spotL.length(); i++) {
	if (spotL[i].color == vec3(0)) continue;
	float angle = dot(-spotL[i].direction, normalize(spotL[i].position - pos));
	if (angle < 0.0f) angle = 0.0f;
	if (angle > spotL[i].innerAngle) {
	    angle = 1.0f;
	} else if (angle < spotL[i].outerAngle) {
	    angle = 0.0f;
	} else {
	    angle = 1 - (angle - spotL[i].innerAngle) / (spotL[i].outerAngle - spotL[i].innerAngle);
	}
    float shadowFactor = texture(gShadowSpot, v_uv).r;
	fragColor.rgb += phong(n, (spotL[i].position - pos), v, spotL[i].color * diffuse, kd * angle,
			       spotL[i].color * specular, ks * angle, shininess, true, spotL[i].attenuation) * shadowFactor;
    }
}
