#version 430 core

#define MAX_DLIGHTS 2
#define MAX_PLIGHTS 10
#define MAX_SLIGHTS 32

in vec3 v_position;
in vec3 v_normals;
in vec2 v_uv;
in vec4 v_lightSpacePos;
out vec4 fragColor;

uniform sampler2D shadowMap;

uniform struct object_material {
    sampler2D ambient;
    sampler2D diffuse;
    sampler2D specular;

	vec3 ka, kd, ks, ke;

    float shininess;
    float shininessStrength;
} material;

uniform float brightness;
uniform vec3 objColor;
uniform vec3 camera_world;

uniform struct directional_light {
	vec3 color;
	vec3 direction;
} dirL[MAX_DLIGHTS];

uniform struct point_light {
	vec3 color;
	vec3 position;
	vec3 attenuation;
} pointL[MAX_PLIGHTS];

uniform struct spot_light {
	vec3 color;
	vec3 position;
	vec3 attenuation;
	vec3 direction;
	float innerAngle;
	float outerAngle;
} spotL[MAX_SLIGHTS];


float calcShadowFactor(vec4 lightSpacePos) {
	vec3 projCoords = lightSpacePos.xyz / lightSpacePos.w;
	vec2 uvCoords;
	uvCoords.x = 0.5 * projCoords.x + 0.5;
	uvCoords.y = 0.5 * projCoords.y + 0.5;
	float z = 0.5 * projCoords.z + 0.5;
	float depth = texture(shadowMap, uvCoords).x;
	if (depth < z) {
		return 0.5;
	} else {
		return 1;
	}
}

vec3 phong(vec3 n, vec3 l, vec3 v, vec3 diffuseC, vec3 kd, vec3 specularC, vec3 ks, float shininess, bool attenuate, vec3 attenuation) {
	float d = length(l);
	l = normalize(l);
	float att = 1.0;
	if (attenuate) att = 1.0f / (attenuation.x + d * attenuation.y + d * d * attenuation.z);
	vec3 r = reflect(-l, n);
	return (kd * diffuseC * max(0, dot(n, l)) + ks * specularC * pow(max(0, dot(r, v)), shininess)) * att;
}

void main()
{
	vec3 n = normalize(v_normals);
	vec3 v = normalize(camera_world - v_position);

	fragColor = vec4(texture(material.diffuse, v_uv).rgb * 0.02f, 1); // ambient light
	float shadowFactor = calcShadowFactor(v_lightSpacePos);

	// add directional light contribution
	for (uint i = 0; i < dirL.length(); i++) {
	    if (dirL[i].color == vec3(0)) continue;
	    fragColor.rgb += phong(n, -dirL[i].direction, v, dirL[i].color * brightness * texture(material.diffuse, v_uv).rgb, material.kd,
				   dirL[i].color * texture(material.specular, v_uv).rgb, material.ks, material.shininess, false, vec3(0)) * shadowFactor + material.ke;
	}

	// add point light contribution
	for (uint i = 0; i < pointL.length(); i++) {
		if (pointL[i].color == vec3(0)) continue;
		fragColor.rgb += phong(n, (pointL[i].position - v_position), v, pointL[i].color * texture(material.diffuse, v_uv).rgb, material.kd,
			pointL[i].color * texture(material.specular, v_uv).rgb, material.ks, material.shininess, true, pointL[i].attenuation) * shadowFactor + material.ke;
	}

	// add spot light contribution
	for (uint i = 0; i < spotL.length(); i++) {
		if (spotL[i].color == vec3(0)) continue;
		float angle = dot(-spotL[i].direction, normalize(spotL[i].position - v_position));
		if (angle < 0.0f) angle = 0.0f;
		if (angle > spotL[i].innerAngle) {
			angle = 1.0f;
		} else if (angle < spotL[i].outerAngle) {
			angle = 0.0f;
		} else {
			angle = 1 - (angle - spotL[i].innerAngle) / (spotL[i].outerAngle - spotL[i].innerAngle);
		}
		fragColor.rgb += phong(n, (spotL[i].position - v_position), v, spotL[i].color * texture(material.diffuse, v_uv).rgb, material.kd * angle,
			spotL[i].color * texture(material.specular, v_uv).rgb, material.ks * angle, material.shininess, true, spotL[i].attenuation) * shadowFactor + material.ke;
	}
}
