#version 330 core

layout(location = 0) out vec4 fragColor;

in vec3 o_position;
in vec2 o_textCoord;
in mat3 o_tbnMat;

struct Material {
	vec3 diffuse;
	vec3 specular;
	vec3 emissive; //for self-illuminating objects
	float shininess;
};

struct DirectLight {
	vec3 direction;

	vec3 diffuse;
	vec3 specular;
	vec3 ambient;
};

struct PointLight {
	vec3 position;

	vec3 diffuse;
	vec3 specular;
	vec3 ambient;

	float linear;
	float quadratic;
	float constant;
};

uniform Material u_material;
uniform DirectLight u_directLight;
uniform PointLight u_pointLight;
uniform sampler2D u_diffuseMap;
uniform sampler2D u_normalMap;
uniform sampler2D u_specularMap;
uniform sampler2D u_depthMap;

void main() {
	// need all illuminations
	vec3 viewVec = normalize(transpose(o_tbnMat) * -o_position); // Vector from fragment to camera (camera always at 0, 0, 0)

	// offset texture coordinates for parallax mapping
	//float depth = texture(u_depthMap, o_textCoord).r; //depth from depthmap and texture coordinate from current fragement
	// large displacement with nearly parallel viewVec, as with perpendicular: viewVec.xy / viewVec.z
	//vec2 displacement = viewVec.xy / viewVec.z * (depth * 0.1); //heightScale  = 0.1
	//vec2 newTextCoord = o_textCoord - displacement;
	//if (newTextCoord.x > 1.0 || newTextCoord.y > 1.0 || newTextCoord.x < 0.0 || newTextCoord.y < 0.0) discard;

	// Parallax Occlusion Mapping: track view vector and test against depth layers
	float minLayer = 8;
	float maxLayer = 32;
	float layerNumber = mix(maxLayer, minLayer, abs(dot(vec3(0.0, 0.0, 1.0), viewVec)));// linearly interpolate between maxLayer and minLayer
	float layerStep = 1.0/layerNumber;

	// large displacement with nearly parallel viewVec, as with perpendicular: viewVec.xy / viewVec.z
	vec2 dispDirection = viewVec.xy / viewVec.z * 0.1; // heightScale  = 0.1
	vec2 shiftTextCoord = dispDirection / layerNumber; // per layer

	vec2 curTextCoord = o_textCoord;
	float curDepth = texture(u_depthMap, o_textCoord).r; // depth from depthmap and texture coordinates from current fragement
	float currentStep = 0.0; // start top down

	while (currentStep < curDepth) {
		curTextCoord -= shiftTextCoord; // shift along direction
		curDepth = texture(u_depthMap, curTextCoord).r; // depth from current texture coordinates
		currentStep += layerStep;
	}

	float preDepth = curDepth - currentStep; // depth before traverse
	vec2 preTextCoord = curTextCoord + shiftTextCoord; // before traverse
	float postDepth = texture(u_depthMap, preTextCoord).r - currentStep + layerStep;// depth after traverse

	float weight = preDepth / (preDepth - postDepth);
	vec2 newTextCoord = weight * preTextCoord + (1.0 - weight) * curTextCoord;

	// normal from normal map
	vec3 normal = texture(u_normalMap, newTextCoord).rgb;
	normal = normalize(normal * 2.0 - 1.0);
	normal = normalize(o_tbnMat * normal);

	// texture
	vec4 diffuseColor = texture(u_diffuseMap, newTextCoord);
	vec4 specularIntensity = texture(u_specularMap, newTextCoord);

	// direction illumination
	vec3 light = normalize(-u_directLight.direction);
	vec3 reflection = reflect(-light, normal);
	vec3 ambient = u_directLight.ambient * diffuseColor.xyz;
	vec3 diffuse = u_directLight.diffuse * max(dot(normal, light), 0.0) * diffuseColor.xyz;
	vec3 specular = u_directLight.specular * pow(max(dot(reflection, viewVec), 0.000001), u_material.shininess) * u_material.specular * specularIntensity.xyz;

	// point illumination (http://wiki.ogre3d.org/-Point+Light+Attenuation)
	vec3 pointLightDirection = normalize(u_pointLight.position - o_position);
	reflection = reflect(-pointLightDirection, normal);
	float dist = length(u_pointLight.position - o_position);
	float attenuation = 1.0 / (u_pointLight.constant + u_pointLight.linear * dist + u_pointLight.quadratic * pow(dist,2.0f));
	ambient += attenuation * u_pointLight.ambient * diffuseColor.xyz;
	diffuse += u_pointLight.diffuse * max(dot(normal, pointLightDirection), 0.0) * diffuseColor.xyz;
	specular += attenuation * u_pointLight.specular * pow(max(dot(reflection, viewVec), 0.00000001), u_material.shininess) * u_material.specular * specularIntensity.xyz;

	fragColor = vec4(ambient + diffuse + specular + u_material.emissive, 1.0);
}