#version 330 core
out vec4 FragColor;

in vec2 TexCoords;

uniform sampler2D gPosition;
uniform sampler2D gNormal;
uniform sampler2D gAlbedoSpec;
uniform sampler2D ssao;
uniform samplerCube depthCubeMap;
uniform float farPlane;
uniform mat4 viewMatrix;

struct PointLight {
    vec3 Position;
    vec3 Color;

    float Linear;
    float Quadratic;
};

uniform PointLight pointLight;

// array of offset direction for sampling
vec3 gridSamplingDisk[20] = vec3[]
(
    vec3(1, 1,  1), vec3( 1, -1,  1), vec3(-1, -1,  1), vec3(-1, 1,  1),
    vec3(1, 1, -1), vec3( 1, -1, -1), vec3(-1, -1, -1), vec3(-1, 1, -1),
    vec3(1, 1,  0), vec3( 1, -1,  0), vec3(-1, -1,  0), vec3(-1, 1,  0),
    vec3(1, 0,  1), vec3(-1,  0,  1), vec3( 1,  0, -1), vec3(-1, 0, -1),
    vec3(0, 1,  1), vec3( 0, -1,  1), vec3( 0, -1, -1), vec3( 0, 1, -1)
);

float CreateShadow(vec3 fragPos) {
    float shadow = 0.0;
    float bias = 0.15;
    int samples = 20;

    vec3 viewToLightDirection = fragPos - vec3(inverse(viewMatrix) * vec4(pointLight.Position, 1.0));
    float currentDepth = length(viewToLightDirection);

    vec3 cameraPos = vec3(viewMatrix[0][3],viewMatrix[1][3],viewMatrix[2][3]); // opengl matrix access is as mat[col][row]
    float cameraDistance = length(cameraPos - fragPos);
    float diskRadius = (1.0 + (cameraDistance / farPlane)) / 25.0;

    for(int i = 0; i < samples; ++i)
    {
        float nearestDepth = texture(depthCubeMap, viewToLightDirection + gridSamplingDisk[i] * diskRadius).r;
        nearestDepth *= farPlane;   // range between 0 and farPlane
        if(currentDepth - bias > nearestDepth) {
            shadow += 1.0;
        }
    }
    shadow /= float(samples);
    return shadow;

    // uncomment to show shadow depth
    // return nearestDepth;
}

void main()
{
    // retrieve data from gbuffer
    vec3 FragPos = texture(gPosition, TexCoords).rgb;
    vec3 Normal = texture(gNormal, TexCoords).rgb;
    vec3 Diffuse = texture(gAlbedoSpec, TexCoords).rgb;
    float specularIntensity = texture(gAlbedoSpec, TexCoords).a;
    float AmbientOcclusion = texture(ssao, TexCoords).r;

    // then calculate lighting as usual
    vec3 ambient = vec3(0.3 * Diffuse * AmbientOcclusion);
    vec3 lighting  = ambient;
    vec3 viewDir  = normalize(-FragPos); // viewpos is (0.0.0)
    // diffuse
    vec3 lightDir = normalize(pointLight.Position - FragPos);

    vec3 fragInWorldSpace = vec3(inverse(viewMatrix) * vec4(FragPos, 1.0));
    float shadow = CreateShadow(fragInWorldSpace);
    vec3 diffuse = max(dot(Normal, lightDir), 0.0) * Diffuse * pointLight.Color;
    // specular
    vec3 halfwayDir = normalize(lightDir + viewDir);
    float spec = pow(max(dot(Normal, halfwayDir), 0.0), 8.0);
    vec3 specular = pointLight.Color * spec * specularIntensity;
    // attenuation
    float distance = length(pointLight.Position - FragPos);
    float attenuation = 1.0 / (1.0 + pointLight.Linear * distance + pointLight.Quadratic * distance * distance);
    diffuse *= attenuation * (1.0 - shadow);
    specular *= attenuation * (1.0 - shadow);
	/*float shadow = CreateShadow(FragPos);
    vec3 lighting = (ambient + (1.0 - shadow) * (diffuse + specular)) * Diffuse;*/
    lighting += diffuse + specular;

    FragColor = vec4(lighting, 1.0);
    // FragColor = vec4(vec3(shadow / farPlane), 1.0);
}