#version 330 core
layout (location = 0) out vec3 gPosition;
layout (location = 1) out vec3 gNormal;
layout (location = 2) out vec4 gAlbedoSpec;

in vec2 TexCoords;
in vec3 FragPos;
in mat3 vert_tbnMat;

uniform sampler2D texture_diffuse1;
uniform sampler2D texture_normal1;
uniform sampler2D texture_specular1;
uniform sampler2D texture_depth1;

struct Material {
    vec3 diffuse;
    vec3 specular;
    vec3 emissive;
    float shininess;
};

uniform Material material;

vec2 ParallaxMapping(vec2 texCoords, vec3 viewDir) {
    // Parallax Occlusion Mapping: track view vector and test against depth layers
    float minLayer = 8;
    float maxLayer = 32;
    float layerNumber = mix(maxLayer, minLayer, abs(dot(vec3(0.0, 0.0, 1.0), viewDir))); // linearly interpolate between maxLayer and minLayer
    float layerStep = 1.0 / layerNumber;

    // large displacement with nearly parallel viewDir, as with perpendicular: viewDir.xy / viewDir.z
    vec2 dispDirection = viewDir.xy / viewDir.z * 0.1; // heightScale  = 0.1
    vec2 shiftTextCoord = dispDirection / layerNumber; // per layer

    vec2 curTextCoord = texCoords;
    float curDepth = texture(texture_depth1, texCoords).r; // depth from depthmap and texture coordinates from current fragement
    float currentStep = 0.0; // start top down

    while (currentStep < curDepth) {
        curTextCoord -= shiftTextCoord; // shift along direction
        curDepth = texture(texture_depth1, curTextCoord).r; // depth from current texture coordinates
        currentStep += layerStep;
    }

    float preDepth = curDepth - currentStep; // depth before traverse
    vec2 preTextCoord = curTextCoord + shiftTextCoord; // before traverse
    float postDepth = texture(texture_depth1, preTextCoord).r - currentStep + layerStep;// depth after traverse

    float weight = preDepth / (preDepth - postDepth);
    return  weight * preTextCoord + (1.0 - weight) * curTextCoord;
}

void main()
{
    // store the fragment position vector in the first gbuffer texture
    gPosition = FragPos;

    // parallax mapping
    vec3 viewVec = normalize(transpose(vert_tbnMat) * -FragPos); // here the transposed tbn matrix is used, so we convert from world space to tangent space
    vec2 newTextCoord = ParallaxMapping(TexCoords, viewVec);

    // also store the per-fragment normals into the gbuffer
    // normal from normal map
    gNormal = normalize(vert_tbnMat * (2.0 * texture(texture_normal1, newTextCoord).rgb - 1.0)); // this tbn converts from tangent space to world space

    // and the diffuse per-fragment color
    gAlbedoSpec.rgb = texture(texture_diffuse1, newTextCoord).rgb + material.emissive;
    // store specular intensity in gAlbedoSpec's alpha component
    gAlbedoSpec.a = texture(texture_specular1, newTextCoord).r;
}