﻿#version 430 core

in vec2 interpolatedUVCoordinate;
in vec3	interpolatedNormal;
in vec3	interpolatedLightDirection;
in vec3	interpolatedCameraDirection;
in vec3 interpolatedFragPos;

uniform sampler2D diffuseTexture;
uniform samplerCube depthMap;

uniform vec3 materialAmbientReflectance;
uniform vec3 materialDiffuseReflectance;
uniform vec3 materialSpecularReflectance; 
uniform float materialShininess;
uniform vec3 lightAmbientIntensity;
uniform vec3 lightDiffuseIntensity; 
uniform vec3 lightSpecularIntensity; 

// for shadows
uniform vec3 lightPos;
uniform float far_plane;

uniform float fft;

out vec4 outColor;

vec3 ambientLighting()
{
   return materialAmbientReflectance * lightAmbientIntensity;
}

vec3 diffuseLighting(in vec3 N, in vec3 L)
{
   // lambertian reflection
   float diffuse = clamp(dot(N, L), 0, 1) ;
   return materialDiffuseReflectance * lightDiffuseIntensity * diffuse;
}


vec3 specularLighting(in vec3 N, in vec3 L, in vec3 V)
{
   float specular = 0;
   // specular reflection if surface is oriented to light source
   if(dot(N, L) > 0)
   {
      vec3 H = normalize(L + V); // half vector
      specular = pow(dot(N, H), materialShininess);
   }
   return materialSpecularReflectance * lightSpecularIntensity * specular;
}

float shadowCalculation(vec3 fragPos)
{
    // get vector between fragment position and light position
    vec3 fragToLight = fragPos - lightPos;
    // use the light to fragment vector to sample from the depth map    
    float closestDepth = texture(depthMap, fragToLight).r;
    // it is currently in linear range between [0,1]. Re-transform back to original value
    closestDepth *= far_plane;
    // now get current linear depth as the length between the fragment and light position
    float currentDepth = length(fragToLight);
    // now test for shadows
    float bias = 0.05; 
    float shadow = currentDepth -  bias > closestDepth ? 1.0 : 0.0;

    return shadow;
} 

void main(void)
{
	// after interpolation normalization
	vec3 L = normalize(interpolatedLightDirection);
	vec3 V = normalize(interpolatedCameraDirection);
	vec3 N = normalize(interpolatedNormal);

	vec3 ambientIntensity = ambientLighting();
	vec3 diffuseIntensity = diffuseLighting(N, L);
	vec3 specularIntensity = specularLighting(N, L, V);

	vec3 diffuseColor = texture(diffuseTexture, interpolatedUVCoordinate).rgb * fft;

	float shadow = shadowCalculation(interpolatedFragPos);  

	outColor.xyz = diffuseColor * (ambientIntensity + diffuseIntensity + specularIntensity);//  * (1.0 - shadow);
	outColor.a = 1.0;

	outColor.xyz *= (1.0 - shadow*0.9);
}