﻿#version 420
##includeCommonVertFunctions##

//PHYSICS
uniform sampler2D prevPosTex;
uniform sampler2D prevVelTex;

uniform sampler3D velFieldTex;

uniform int texWidth;

//BOIDS
uniform sampler2D nnCountTex;
uniform sampler2D nnVelTex;
uniform sampler2D nnPosTex;

uniform int nnTexWidth;
uniform int cubeRes;

uniform vec3 targetPos;

const float v1Strength = 30.000;
const float v2Strength = .01;
const float v3Strength = 0.3;
const float repellRad = 0.003;
const float attractRad = 0.1;
const float conformRad = 0.1;

const float targetAttrStrength = 1.5;

//TIME
uniform float time;
uniform float timeDelta;
uniform float simStartTime;

//COLLISION
const int MAX_SPHERE_COUNT = 13;

const float SPHERE_BOUNCE = 0.7;
const float WALL_BOUNCE = 0.5;

uniform int sphere_cnt;
uniform vec3 spherePositions[MAX_SPHERE_COUNT];
uniform float sphereRadii[MAX_SPHERE_COUNT];
uniform float walls[6];

//OUT
out vec3 outPos;
out vec3 outVel;

//BOIDS FUNCTIONS

vec3 boidsV1Attract(vec3 nnPos, vec3 pos, float attractRadius){
	vec3 diff = (nnPos - pos);
	if(length(diff) < attractRadius){
		return diff  * v1Strength;
	}
	return vec3(0.0);
}

vec3 boidsV2Repell(vec3 nnPos, vec3 pos, float repellRadius){
	vec3 diff = (pos - nnPos);
	float mag = length(diff);
	if(mag < repellRadius){
		mag /= repellRadius;
		mag = 1.0/(mag+ 0.1*repellRadius);
		diff = normalize(diff);
		return diff   * v2Strength;
	}
	return vec3(0.0);
}

vec3 boidsV3ConformVel(vec3 nnPos, vec3 pos, vec3 nnVel, vec3 vel, float conformRadius){
	vec3 diff = (pos - nnPos);
	if(length(diff) < conformRadius){
			return nnVel * v3Strength;
	}
	return vec3(0.0);
}

//COLLISION FUNCTIONS
int didCollideWithSphere(vec3 sP, float sR, vec3 P) {
    float diff = distance(sP, P);
    if(diff  < sR){
        return 1;
    } else {
        return 0;
    }
}

int didCollideWithWall(vec3 P) {
	if(P.x < walls[0]){
		return 0;
	} else if(P.x > walls[1]){
		return 1;
	} else if(P.y < walls[2]){
		return 2;
	} else if(P.y > walls[3]){
		return 3;
	} else if(P.z < walls[4]){
		return 4;
	} else if(P.z > walls[5]){
		return 5;
	} else {
		return -1;
	}
}

vec3 sphereCollisionDeflectVel(vec3 oldPos, vec3 newPos, vec3 sP){
    vec3 impactPos = (newPos + oldPos) / 2.0; //needs exact line/sphere intersection maybe??
    vec3 impactVec = newPos - oldPos;
    vec3 impactNorm = normalize(impactPos - sP);
    
    vec3 refl = -reflect(-impactVec, impactNorm);
    return refl;
}

vec3 wallCollisionDeflectVel(vec3 oldPos, vec3 newPos, int wallCollision){
	vec3 impactVec = newPos - oldPos;
	vec3 impactNormal = vec3(0.0);
	if(wallCollision == 0){
		impactNormal.x = 1.0;
	} else if(wallCollision == 1){
		impactNormal.x = -1.0;
	} else if(wallCollision == 2){
		impactNormal.y = 1.0;
	} else if(wallCollision == 3){
		impactNormal.y = -1.0;
	} else if(wallCollision == 4){
		impactNormal.z = 1.0;
	} else if(wallCollision == 5){
		impactNormal.z = -1.0;
	}
	vec3 refl = reflect(impactVec, impactNormal);
	return refl;
}

vec3 wallCollisionDeflectVel(vec3 oldPos, vec3 newPos, vec3 wallNorm){
    vec3 impactPos = (newPos + oldPos) / 2.0; //needs exact line/plane intersection
    vec3 impactVec = newPos - oldPos;
    
    vec3 reflection = -reflect(-impactVec, normalize(wallNorm));
    return reflection;
}

// PARTICLE SPAWNING
vec3 initialPosition(vec3 boxSize, vec3 boxPos){
	int cubeWidth = int(ceil(pow((texWidth*texWidth), 1.0/3.0)));
	int vid = gl_VertexID;
	
	int planeSz = cubeWidth*cubeWidth;

	int z = vid / planeSz;
	int rem = vid - z * planeSz;
	int y = rem / cubeWidth;
	rem = rem - y * cubeWidth;
	int x = rem;

	float xf = float(x);
	float yf = float(y);
	float zf = float(z);

	return (vec3(xf, yf, zf)/float(cubeWidth) - vec3(0.5)) * boxSize + boxPos;
}

//UTILITY FOR PER VERTEX VARIATION
float vIDModulation(){
	float mod = sin(gl_VertexID) * 0.5 + 0.5;
	return mod;
}

//VECTORVIELD LOOKUP
vec3 vecFieldVel(vec3 oldPos, vec3 vecFieldOffset){
	vec3 refPos = oldPos - vecFieldOffset;
	if(abs(refPos.x) > 0.5 ||
	abs(refPos.y) > 0.5 ||
	abs(refPos.z) > 0.5)
	{
		return vec3(0.0);	
	}

	return texture(velFieldTex, (refPos + vec3(0.5))).rgb;
}

//TARGET ATTRACTION TESTING STUFF
vec3 blendWithTestingTarget(vec3 oldPos, vec3 velField){
	float timeBase = time + float(1+gl_VertexID*3);
	float zeroMinTimeSin = sin(timeBase) * 0.5 + 0.5;
	float a = zeroMinTimeSin * 2.0 - 1.0;
	
	vec3 targetPos = vec3(sin(timeBase), cos(time), -sin(time));
	vec3 targetForceVel = targetPos - oldPos;

	vec3 velUpdate = velField*a + (1.0-a) * targetForceVel;
	return velUpdate;
}

vec3 getTargetAttraction(vec3 oldPos, float speed){
	float timeBase = time * speed;
	vec3 targetForceVel = targetPos - oldPos;

	return targetForceVel;
}

void main() {

	vec2 texPos = getPhysTextureSpacePosition(gl_VertexID, texWidth);  
	vec2 renderPos = texPos  * 2.0 - vec2(1.0);

	vec3 oldPos;
	vec3 oldVel;
	if(time < simStartTime){
		oldPos = initialPosition(vec3(0.3, 0.2, 0.3), vec3(0, 0.2, 0));
		oldVel = vec3(0.0);
	} else {
		oldPos = texture(prevPosTex, texPos).rgb;
		oldVel = texture(prevVelTex, texPos).rgb;
	}

	vec2 nnTexPos = getNNGridTexCoords(oldPos, vec3(-0.5, 0, -0.5), vec3(1.0, 1.0, 1.0), cubeRes, nnTexWidth);

	float neighbourCount = texture(nnCountTex, nnTexPos).r;
	vec3 nnVel =  texture(nnVelTex, nnTexPos).rgb / neighbourCount;
	vec3 nnPos =  texture(nnPosTex, nnTexPos).rgb / neighbourCount;

	float simTime = time - simStartTime;

	vec3 velUpdate = getTargetAttraction(oldPos, .5) * targetAttrStrength;
	//TESTING
	//if(mod(gl_VertexID, 20) == 0)
	//	velUpdate *= 0.3;

	//BOIDS UPDATE
	if(simTime > 4.0 && neighbourCount > 10.0){
		velUpdate = velUpdate 
					+ boidsV1Attract(nnPos, oldPos, attractRad)
					+ boidsV2Repell(nnPos, oldPos, repellRad) * neighbourCount
					+ boidsV3ConformVel(nnPos, oldPos, nnVel, oldVel, conformRad);
	}


	vec3 newVel = oldVel + velUpdate * timeDelta ;

	vec3 velField = vecFieldVel(oldPos, vec3(0, 0.5, 0)) * 0.2;
	if(simTime < 2.0)
		newVel = velField;

	vec3 newPos = oldPos + newVel * timeDelta;

	for(int i=0; i<sphere_cnt; i++){
		vec3 sPos = spherePositions[i];
		float sRad = sphereRadii[i];
		if(didCollideWithSphere(sPos, sRad, newPos) == 1){
			newVel = SPHERE_BOUNCE * sphereCollisionDeflectVel(oldPos, newPos, sPos) / timeDelta; //Velocity is calculated from previous velocity-based position update that had been multiplied by timeDelta, so to get correct vel. we have to divide
			newPos = oldPos + newVel * timeDelta; // Position update again needs time Delta influence.
		}
	}

	int wallCollision = didCollideWithWall(newPos);
	if(wallCollision >=0){
		newVel = WALL_BOUNCE * wallCollisionDeflectVel(oldPos, newPos, wallCollision) / timeDelta;
		newPos = oldPos + newVel * timeDelta;
	}

	newVel *= 1.0 - 0.5 * timeDelta;//  * (vIDModulation()*0.5+0.5); // DAMPING


	outPos = newPos;
	outVel = newVel;

	// transform to according position
	gl_Position = vec4(renderPos, 0, 1.0);
	//gl_Position = vec4(float(gl_VertexID/100.0), 0, 0, 1.0);
}