#version 430 core
/*
* Copyright 2010 Vienna University of Technology.
* Institute of Computer Graphics and Algorithms.
* This file is part of the ECG Lab Framework and must not be redistributed.
*/

in VertexData {
	vec3 position_world;
	vec3 normal_world;
	vec2 uv;
	vec2 uv_distorted;
	vec4 clipSpace;
	vec3 toCameraVector;
} vert;

layout (location = 0) out vec4 color;
layout (location = 1) out vec4 BrightColor;

uniform vec3 camera_world;
uniform vec3 cameraDir;
uniform float moveFactor;
uniform sampler2D diffuseTexture;
uniform sampler2D reflectionTexture;
uniform sampler2D refractionTexture;
uniform sampler2D dudvMap;
uniform sampler2D normalMap;

const vec4 blueColor = vec4(0.0, 0.8, 1.0, 1.0);
const float waveStrength = 0.02;
const float shineDamper = 20.0;
const float reflectivity = 0.6;

uniform struct DirectionalLight {
	vec3 color;
	vec3 direction;
} dirL;

uniform struct PointLight {
	vec3 color;
	vec3 position;
	vec3 attenuation;
} pointL;


void main() {
	// convert to screen space coordinates
	vec2 ndc = (vert.clipSpace.xy/vert.clipSpace.w)/2.0 + 0.5;
	vec2 reflectTexCoords = vec2(ndc.x, -ndc.y);
	vec2 refractTexCoords = vec2(ndc.x, ndc.y);

	// sample dudv map, convert it to [-1, 1]
	vec2 distortion = (texture(dudvMap, vec2(vert.uv.x + moveFactor, vert.uv.y)).rg * 0.1);
	distortion = vert.uv + vec2(distortion.x, distortion.y + moveFactor);
	vec2 totalDistortion = (texture(dudvMap, distortion).rg * 2.0 - 1.0) * waveStrength;

	reflectTexCoords += totalDistortion;
	refractTexCoords += totalDistortion;
	vec2 distortedTex = vert.uv + totalDistortion;

	reflectTexCoords.x = clamp(reflectTexCoords.x, 0.001, 0.999);
	reflectTexCoords.y = clamp(reflectTexCoords.y, -0.999, -0.001);
	refractTexCoords = clamp(refractTexCoords, 0.001, 0.999);

	// sample reflection and refraction texture and mix them
	vec4 reflectColor = texture(reflectionTexture, reflectTexCoords);
	vec4 refractColor = texture(refractionTexture, refractTexCoords);
	vec4 texColor = texture(diffuseTexture, distortedTex);

	vec4 normalMapColor = texture(normalMap, distortion);
	vec3 normal = vec3(normalMapColor.r * 2.0 - 1.0, normalMapColor.b, normalMapColor.g * 2.0 - 1.0);
	normal = normalize(normal);

	vec3 viewVector = normalize(-cameraDir);
	float refractiveFactor = dot(viewVector, vec3(0, 1, 0));
	refractiveFactor = pow(refractiveFactor, 10.9);
	refractiveFactor = clamp(refractiveFactor, 0.001, 0.999);

	vec3 reflectedLight = reflect(normalize(dirL.direction), normal);
	float specular = max(dot(reflectedLight, viewVector), 0.0);
	specular = pow(specular, shineDamper);
	vec3 specularHighlights = dirL.color * specular * reflectivity;

	color = mix(reflectColor, refractColor, refractiveFactor);
	color = mix(color, texColor, 0.45);
	color = mix(color, blueColor, 0.15);
	color += vec4(specularHighlights, 0.0);
	//color = refractColor;

	// check whether fragment output is higher than threshold, if so output as brightness color
    float brightness = dot(color.rgb, vec3(0.0, 1.1, 0.0));
    if(color.y > color.z && color.y > color.x) {
        BrightColor = vec4(color.rgb, 1.0);
    } else {
        BrightColor = vec4(0.0, 0.0, 0.0, 1.0);
	}
}

