#pragma once

#include <windows.h>
#include <glew.h>
#include <gl\gl.h>
#include <GL\glut.h>
#include <math.h>
#include "Data.h"
#include "planeBaseNavigator.h"

class planeAdmin;
class GLWidget;

#include <vector>
using namespace std;


struct Point {
	float x;
	float y;
	float z;
};

/**
* Base class for every visualization type.
*/
class planeBase {
public:
	/** Constructor */
	planeBase();

	/** Destructor */
	~planeBase();

	/** initialize plane with data 
		 @param cols choosen columns that will be displayed
		 @param dataSet loaded data 
		 @param _parentWidget the widget in which the plane gets displayed*/
	void init(vector<int> cols, Data& dataSet, GLWidget *_parentWidget);

	/** initialize plane */
	void init();

	/** renders the plane
		@param dT deltaTime to the previous frame */
	void render(float dT);

	/** renders the plane transparent for the z-Buffer */
	void renderDepth();

	/** checks if mouseclick happend on plane
		@param x x position of the mouse cursor
		@param y y position of the mouse cursor
		@return true, if click is on the plane, false if click is outside of the plane */
	bool mouseSelectClick(float x, float y);

	/** deselects all Nodes of the plane */
	void mouseDeselect();

	/** checks mouse movement while the mousebutton is pressed
		@param x x position of the mouse cursor
		@param y y position of the mouse cursor */
	void mouseNavigateDrag(float x, float y);

	/** indicates that the mouse button is now clicked
		@param x x position of the mouse cursor
		@param y y position of the mouse cursor
		@param button the index of the mousebutton 
		@return  true, if click is on the plane, false if click is outside of the plane */
	bool mouseNavigateClickOn(float x, float y, int button=0);

	/** indicates taht the mouse button is now released */
	void mouseNavigateClickOff();

	/** displays the navigators of the plane */
	void showNavigators();

	/** hides the navigators of the plane */
	void hideNavigators();

	/** sets the position of the plane
		@param x x position
		@param y y position
		@param z z position */
	void setPos(float x, float y, float z);

	/** sets the orientation of the plane
		@param x rotation around x-axis
		@param y rotation around y-axis */
	void setRot(float x, float y);

	/** returns the z-Value of the plane
		@return z-Value of the plane */
	float getDepth();

	/** returns the point with the given id
		@param _id index of the point
		@return position of the point */
	Point getPoint(int _id);

	/** returns the middle of the plane
		@return middle of the plane */
	Point getMiddle();

	/** returns the index of the selected point
		@return index of the selected point */
	int getSelectedPoint();

	/** indicates if the plane has to be redrawn */
	bool needsRenderUpdate;

	/** type of the plane */
	int type;

	/** indicates if the plane has to be deleted */
	bool deleteMe;

	/** pointer to the planeAdmin */
	planeAdmin *masterPlane;

	/** starts animation to 2D view */
	void animateTo2D();

	/** starts animation to 3D view */
	void animateTo3D();

	/** retursn the current view state */
	int getCur2D3Dstate();

private:
	/** creates the framebuffer object */
	void createFBO();

	/** destroys the framebuffer object*/
	void destroyFBO();

	/** animation between 2D and 3D view
		@param dT delta Time */
	void animate2D3D(float dT);

	/** transforms the screencoordinates to planecoordinates
		@param x the x coordinate on screen
		@param y the y coordinate on screen */
	void screenToPlane(float& x, float& y);

	/** initialize plane with data 
		 @param cols choosen columns that will be displayed
		 @param dataSet loaded data */
	virtual void initPlane(vector<int> cols, Data& dataSet) {};

	/** initialize plane */
	virtual void initPlane() {};

	/** renders the visualization */
	virtual void renderVis()=0;

	/** initialization state of the framebuffer object */
	bool fboInitialised;

	/** initialization state of the plane */
	bool planeInitialised;

	/** position of the plane */
	float posX, posY, posZ;

	/** orientation of the plane */
	float rotX, rotY;

	/** mouseposition for navigation */
	float naviMousePosX, naviMousePosY;

	/** last 3D position */
	float last3DposX, last3DposY, last3DposZ;

	/** last 3D orientation */
	float last3DrotX, last3DrotY;

	/** 2D position */
	float dest2DposX, dest2DposY, dest2DposZ;

	/** 2D orientation */
	float dest2DrotX, dest2DrotY;

	/** current navigation mode */
	int navigationMode;			// 0... nichts
								// 1... buch
								// 2... garagentor
								// 3... verschieben (x, y)
								// 4... verschieben (z)

	/** book navigator */
	planeBaseNavigator navBook;

	/** garage navigator */
	planeBaseNavigator navGarage;

	/** drag navigator */
	planeBaseNavigator navDrag;

	/** delete navigator */
	planeBaseNavigator navDelete;

	/** zoom navigator */
	planeBaseNavigator navZoom;

	/** renders the navigators
		@param dT delta time */
	void renderNavigators(float dT);

	/** framebuffer object */
	GLuint fbo;

	/** texture for framebuffer object */
	GLuint fboTexture;

	/** true if currently animating to 2D view */
	bool animatingTo2D;

	/** true if currently animating to 3D view */
	bool animatingTo3D;

	/** current animation state */
	int cur2D3Dstate;			// 1... animation laeuft
								// 2... im 2D modus
								// 3... im 3D modus

	/** animation speed */
	float animationSpeed;

	/** alpha value in 3D view */
	float alpha3D;

	/** current alpha value of the plane */
	float alpha;

	/** parent widget */
	GLWidget *parentWidget;

protected:
	/** checks if a node has been selected
		@param x x position on the plane
		@param y y position on the plane */
	virtual void selectClick(float x, float y)=0;

	/** linking points on the plane */
	vector<Point> points;

	/** index of the selected node */
	int selectedPoint;
};