package interring;
/** Ausgabefenster
 * In diesem Fenster wird der Interring dargestellt. Dieses 
 * separate Fenster verwendet die Technologie des Page-Flippings
 * um eine flimmerfreie Darstellung zu gewhrleisten.
 * Weiters werden hier auch alle Mausevents verarbeitet.
 * @author Thomas Kment
 * @author Christof Kier
 */
import javax.swing.*;

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;

import java.awt.*;

import java.awt.Dimension;

import java.awt.image.BufferedImage;
import java.awt.image.BufferStrategy;
import java.awt.geom.*;
import java.io.File;



import java.util.Vector;

public class GUI_Output extends JFrame{
	private BufferStrategy bufferStrategy; // fr Pageflipping
	private BufferedImage bimg;// Zeichenflche
	private GeometryNode geometry;// Geometrieklasse des Interring
	private AffineTransform at;// Transformationen des Inttering
	private ThicknessList thicknesslist; // Dickenliste fr radial Distortion
	
	private GeometryNode selected; // welcher Node ist ausgewhlt?
		
	private double rotation = 0.0; // rotation fr at
	private double scale = 1.0; // skalierung fr at
	private int translation_x = 0; // translation fr at
	private int translation_y = 0; // translation fr at
	
	private boolean mouse_button_left_down = false; // mausstatus
	private boolean mouse_button_right_down = false; // mausstatus
	private boolean distortion_mode = false; // mausstatus - im verzerrungsmodus?
	private boolean inner_edge = false; // mausstatus - innen oder auenkante gewhlt?
	private int old_mouse_x = 0; // mausstatus
	private int old_mouse_y = 0;// mausstatus
	private int mouse_x = 0;// mausstatus
	private int mouse_y = 0;// mausstatus
	
	/** Konstruktor
	 * Initialisierung von Variablen und Objekten.
	 *
	 */
	public GUI_Output(){
		super("IRing - Ausgabefenster");		
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		this.setBackground(Color.white);
		this.setLayout(null);
		//this.setIgnoreRepaint(true);
		//this.setResizable(false);
		this.pack();		
		this.setSize(new Dimension(800,600));		
		this.createBufferStrategy(2);		
		bufferStrategy = this.getBufferStrategy();
		addMouseListener(new MyMouseListener());
		addMouseMotionListener(new MyMouseMotionListener());		
	}
	
	/** Initialisierung des Graphics2D Objektes
	 * 
	 * @param w - Breite
	 * @param h - Hhe
	 * @return Initialisiertes Graphics2D Objekt
	 */
	public Graphics2D createGraphics2D(int w, int h) {
        Graphics2D g2 = null;
        if (bimg == null || bimg.getWidth() != w || bimg.getHeight() != h) {
            bimg = (BufferedImage) createImage(w, h);           
        } 
        g2 = bimg.createGraphics();
        g2.setBackground(getBackground());
        g2.clearRect(0, 0, w, h);     
        return g2;
    }	

	/** Zeichenfunktion des Interring
	 * Wichtig hier ist at - die Matrix  welche alle Transformationen
	 * zusammenfasst. Sie handhabt so Rotation, Translation und Skalierung
	 * des Interring.
	 * @param d Dimension des Fensters
	 * @param g2 Graphics2D Objekt
	 */
	private void drawInterring(Dimension d, Graphics2D g2){
		if(bimg != null){// && rootnode != null){					
			// Matrizen Setup
			at = AffineTransform.getTranslateInstance(d.width/2, d.height/2);
			at.translate(translation_x, translation_y);
			at.scale(scale, scale);
			at.rotate(Math.toRadians(rotation));					
			//
			if(geometry == null){ // Wenn kein Interring vorhanden, zum Testen			
				Line2D one = new Line2D.Double(-40, -40, 40, 40);			
				Line2D two = new Line2D.Double(-40, 40, 40, -40);
				g2.draw(at.createTransformedShape(one));
				g2.draw(at.createTransformedShape(two));
			}else{
				// Aufruf rekursiver Struktur zum Zeichnen des Interrring hier:
				geometry.drawIt(this.thicknesslist, at, g2);
			}
		}
	}

	/** Paint Methode des Fensters
	 * Aufruf der Paint Methode des Java-GUI Elements.
	 */
	public void paint(Graphics g)
	{			
		Dimension d = getSize();        
        Graphics2D g2 = createGraphics2D(d.width, d.height);        

        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        
        BasicStroke stroke = new BasicStroke(0.5f, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_BEVEL);
		g2.setStroke(stroke);
		 
        drawInterring(d, g2);        
        g2.dispose();        
        
        Graphics2D g3 = (Graphics2D)bufferStrategy.getDrawGraphics();

        g3.drawImage(bimg, 0, 0, this);
        
        //
        if(mouse_button_left_down && selected != null){
        	int estimate = selected.getName().length() * 8;        	       	
        	g3.drawRect(this.old_mouse_x-3, this.old_mouse_y-15, estimate, 23);
        	g3.setPaint(new Color(240,220,160,170));
        	g3.fillRect(this.old_mouse_x-3, this.old_mouse_y-15, estimate, 23);
        	g3.setPaint(Color.black);
        	g3.drawString(selected.getName(), this.old_mouse_x,  this.old_mouse_y);        	        	
        }
        
        //
        g3.dispose();        
        bufferStrategy.show();
        
	}
	
	/** Methode zur Generierung der Geometry des Interring
	 * 
	 * @param interring - Datenstruktur
	 */
	public void setGeometry(InterringNode interring){
		// root-node (rest rekursiv)		
		geometry = generateGeometry(null, interring, 0, 0.0, interring.getBranchSize());
		// Thickness List		
		thicknesslist = new ThicknessList(geometry.getMaxDepth()+1, 20.0, 5.0);		
	}	
	
	/** Rekursive Methode zur Erstellung der Geometr
	 * 
	 * @param curTop - Vater des derzeiten Geometrieknotens
	 * @param interring - Derzeit verarbeiteter Knoten
	 * @param distance2root - Tiefe im Baum
	 * @param alpha - Startwinkel fr Ast
	 * @param father_size - Gre des Vaters
	 * @return fertigen Knoten
	 */
	private GeometryNode generateGeometry(GeometryNode curTop, InterringNode interring, int distance2root, double alpha, long father_size){
		double omega = alpha + 360.0 * interring.getBranchSize() / father_size;
		GeometryNode curDir = new GeometryNode(alpha,distance2root,interring.getName(),interring.getBranchSize(),omega);
		//System.out.println("1)Lege Geometrie an: " + distance2root + " alpha:" + alpha + " omega:" + omega );
		//System.out.println(curDir.getName());
		if(curTop != null){
			curTop.addChild(curDir);
		}
		Vector vec = interring.children;
		if(vec.size() > 0){
			for(int i = 0; i < vec.size();i++){
				InterringNode irnode = (InterringNode)vec.get(i);
				if(irnode.getChildCount() > 0){
					generateGeometry(curDir, (InterringNode)vec.get(i), (distance2root+1), alpha, father_size);
					omega = alpha + 360.0 * ((InterringNode)vec.get(i)).getBranchSize() / father_size;
					//System.out.println("2)bergebe Geometrie: " + (distance2root+1) + " alpha:" + alpha + " omega:" + omega );
					//System.out.println(((InterringNode)vec.get(i)).getName());
					alpha = omega;
					
				}else{
					omega = alpha + 360.0 * ((InterringNode)vec.get(i)).getBranchSize() / father_size;
					//System.out.println("Zhler: " + ((InterringNode)vec.get(i)).getBranchSize() + "  Nenner: " + father_size);					
					GeometryNode node = new GeometryNode(alpha, (distance2root+1), ((InterringNode)vec.get(i)).getName(), father_size, omega);
					//System.out.println("3)Lege Geometrie an: " + (distance2root+1) + " alpha:" + alpha + " omega:" + omega );
					//System.out.println(((InterringNode)vec.get(i)).getName());
					curDir.addChild(node);
					alpha = omega;					
				}
			}			
		}
		curDir.setOmega(omega);
		return curDir;
	}
	
	/** Methode fr Picking
	 * 
	 * @param x - Maus X
	 * @param y - Maus Y
	 * @return Knoten, welcher angewhlt wurde
	 */
	private GeometryNode getCandidate(double x, double y){
		Point2D p = new Point2D.Double(x,y);
		Point2D ip = null;
		int distance2root = 0;
		double angle = 0.0;
		try{
			ip = at.inverseTransform(p, null);			
			double dist = Math.sqrt(ip.getX()*ip.getX() + ip.getY()*ip.getY());
			distance2root = thicknesslist.getLayer(dist);
			inner_edge = thicknesslist.isInnerBorder(distance2root, dist);
			x = ip.getX() / dist;
			y = ip.getY() / -dist;			
			angle = Math.toDegrees(Math.acos(x));
			if(y<0.0) angle = 360.0 - angle;			
		}catch(Exception ex){
			
		}
		GeometryNode node = null;
		if(geometry != null && distance2root > -1){
			node = geometry.search(distance2root, angle);
		}
		return node;
	}

	/** Mauseventlistener 
	 * Dieser Listener verarbeitet Ereignisse, welche mit dem Drcken von Maustasten zusammenhngen
	 * @author Thomas Kment
	 * @author Thomas Christof Kier
	 */
	class MyMouseListener
	extends MouseAdapter
	{
		public void mousePressed(MouseEvent event)
		{
			if(event.getButton() == java.awt.event.MouseEvent.BUTTON1){
				mouse_button_left_down = true;
				old_mouse_x = event.getX();
				old_mouse_y = event.getY();			
				
				GeometryNode newselected = getCandidate((double)old_mouse_x, (double)old_mouse_y);
				
				if(selected == null) selected = newselected;
				else{
					if(selected.equals(newselected)){					
						distortion_mode = true;
					}					
					else { 
						selected = newselected;
						distortion_mode = false;
						}
				}
				
				if(selected != null)repaint();
				if(event.getClickCount()>1 && selected != null){
					selected.toggleVisibilityOfChildren();
				} else if (selected != null){
					geometry.clearAllSelections();
					selected.enableSelection();
				} else{
					geometry.clearAllSelections();
				}				
			}
			if(event.getButton() == java.awt.event.MouseEvent.BUTTON3){
				mouse_button_right_down = true;
				old_mouse_x = event.getX();
				old_mouse_y = event.getY();
			}
			

     	}
     	public void mouseReleased(MouseEvent event)
     	{
     		if(event.getButton() == java.awt.event.MouseEvent.BUTTON1){
     			mouse_button_left_down = false;
     			repaint();
			}
			if(event.getButton() == java.awt.event.MouseEvent.BUTTON3){
				mouse_button_right_down = false;
			}
			
   		}
  	}
	
	/** Mauseventlistener 
	 * Dieser Listener verarbeitet Ereignisse, welche mit der Bewegung der Maus zusammenhngen
	 * @author Thomas Kment
	 * @author Thomas Christof Kier
	 */
   class MyMouseMotionListener
   extends MouseMotionAdapter
   {
	   	public void mouseDragged(MouseEvent event)
	   	{
	   	 if(mouse_button_right_down){
	   		mouse_x = event.getX();
	   		mouse_y = event.getY();
	   		
	   		rotation += (mouse_x - old_mouse_x) / 2;
	   		scale -= ((double)mouse_y - (double)old_mouse_y) / 100.0;
	   		if(scale < 0.0) scale = 0.01;
	   		
	   		old_mouse_x = mouse_x;
	   		old_mouse_y = mouse_y;
	   		
	   		repaint();   		 
	   	 }	   	 
	   	 if(mouse_button_left_down){
	   		if(!distortion_mode){
		   		mouse_x = event.getX();
		   		mouse_y = event.getY();
		   		
		   		translation_x += (mouse_x - old_mouse_x);
		   		translation_y += (mouse_y - old_mouse_y);
		   		
		   		old_mouse_x = mouse_x;
		   		old_mouse_y = mouse_y;
		   		
		   		repaint();
	   		}else{
	   			mouse_x = event.getX();
		   		mouse_y = event.getY();
		   		
		   		int diff_y = (mouse_y - old_mouse_y);
		   		int diff_x = (mouse_x - old_mouse_x);
		   		
		   		int len = diff_y - diff_x;
		   		
		   		thicknesslist.modify(selected.getDistance2root(), len, inner_edge);
		   		
		   		
		   		old_mouse_x = mouse_x;
		   		old_mouse_y = mouse_y;
		   		
		   		repaint();
		   		
	   		}
	   	 }
	   	}
   }
}
