/** \file basicLib.glsl
 *  
 *  @author Michael May
 *  @date   22.02.10
 *
 *  Copyright (C) 2009
 *
 *  @brief library of basic defintions and functions
 *  a lot code is based on the book "OpengGL Shading Language Second Edition"
 */
 
#version 140

/// all direction and normals are in eye space
struct LightSourceParameters {
	vec4 ambient;
	vec4 diffuse;
	vec4 specular;
	vec4 position; ///< direction for directional-light
	
	// point- and spotlight
	float constantAttenuation;
	float linearAttenuation;
	float quadraticAttenuation;
	
	// spotlight
	vec3 spotDirection;
	float spotExponent;
	float spotCutoff; ///< (range: [0.0,90.0], 180.0)
	float spotCosCutoff; ///< (range: [1.0,0.0],-1.0)
};

struct Material {
	vec4 ambient;
	vec4 diffuse;
	vec4 specular;
	vec4 emission;
	float shininess;
	bool diffuseTextureAvailable;
};

uniform mat4 projMatrix, viewMatrix, modelMatrix;
uniform mat3 normalMatrix;

uniform sampler2D diffuseTex;

uniform Material FrontMaterial = Material( vec4(.2), vec4(1.), vec4(1.), vec4(0.), .5, false );
uniform float NumEnabledLights;
uniform LightSourceParameters LightSource[ 5 ];
uniform vec3 camPos;
uniform float timer;
uniform vec4 globalAmbient;

//uniform struct Shadow {
//	uniform sampler2D map;
//	float pixelOffset;
//	float epsilon;
//} shadow;


// **************** Lighting ****************

void DirectionalLight( in LightSourceParameters light, in Material material,
						in vec3 ecPosition3, in vec3 normal,
						inout vec3 ambient, inout vec3 diffuse, inout vec3 specular )
{
	vec3 lightDir = normalize(vec3(light.position));
	float nDotL = dot(normal, lightDir);

	if( nDotL > 0. ) { // no diffuse or specular on the dark side of the moon
		// DIFFUSE
		diffuse  += vec3( light.diffuse ) * nDotL;

		// SPECULAR
		if( material.shininess != 0. ) {
			// direction of maximum highlights
			vec3 halfVector = normalize( lightDir - normalize(ecPosition3) );

			float nDotHV = max( 0., dot(normal, halfVector ) );
			specular += vec3( light.specular ) *
				pow( nDotHV, material.shininess );
		}
	}

	// AMBIENT
	ambient  += vec3( light.ambient );
}

void PointLight( in LightSourceParameters light, in Material material,
				in vec3 eye, in vec3 ecPosition3, in vec3 normal,
				inout vec3 ambient, inout vec3 diffuse, inout vec3 specular )
{
	// Compute vector from surface to light position
	vec3 lightDir = vec3(light.position) - ecPosition3;
	float lightDist = length( lightDir );
	lightDir = normalize( lightDir );

	float nDotL = dot(normal, lightDir);

	// Compute attenuation
	float attenuation = 1. / ( light.constantAttenuation +
					light.linearAttenuation * lightDist +
					light.quadraticAttenuation * lightDist * lightDist );

	if( nDotL > 0. ) // no light on the dark side of the moon
	{
		// DIFFUSE
		diffuse  += vec3( light.diffuse ) * attenuation * nDotL;

		// SPECULAR
		if( material.shininess != 0. ) {
			// direction of maximum highlights
			vec3 halfVector = normalize( lightDir + eye );
			float nDotHV = max( 0., dot(normalize(normal), normalize(halfVector) ) );
	
			float pf = pow( nDotHV, material.shininess );
			specular += vec3( light.specular ) * attenuation * pf;
		}
	}

	// AMBIENT
	ambient  += vec3( light.ambient )  * attenuation;
}

void SpotLight( in LightSourceParameters light, in Material material,
				in vec3 eye, in vec3 ecPosition3, in vec3 normal,
				inout vec3 ambient, inout vec3 diffuse, inout vec3 specular )
{
	// Compute vector from surface to light position
	vec3 lightDir = vec3( light.position ) - vec3( ecPosition3 );
	float lightDist = length( lightDir );
	lightDir = normalize( lightDir );
	
	// Compute attenuation
	float attenuation = 1. / ( light.constantAttenuation +
						light.linearAttenuation * lightDist +
						light.quadraticAttenuation * lightDist * lightDist );
	
	// See if point on surface is inside cone of illumination
	float spotDot = dot( -lightDir, normalize(light.spotDirection) );
	
	if( spotDot >= light.spotCosCutoff )
		attenuation *= pow( spotDot, light.spotExponent );
	else attenuation = 0.;

	float nDotL = dot(normal, lightDir);
	if( nDotL > 0. ) // no light on the dark side of the moon
	{
		// DIFFUSE
		diffuse  += vec3( light.diffuse )  * attenuation * nDotL;

		if( material.shininess != 0. ) {
			vec3 halfVector = normalize( lightDir + eye );
			float nDotHV = max( 0., dot(normal, halfVector) );
			float pf = pow( nDotHV, material.shininess );

			// SPECULAR
			specular += vec3( light.specular ) * attenuation * pf;
		}
	}

	// AMBIENT
	ambient  += vec3( light.ambient )  * attenuation;
}

void processLight( in LightSourceParameters light, in vec3 ecPosition3, in vec3 normal,
			in float shadeFactor, // 0=>full shadowed, 1=>not shadowed
			inout vec3 ambient, inout vec3 diffuse, inout vec3 specular )
{
	vec3 eye = vec3( 0., 0., 1. ); // -normalize(ex_ecPosition3)

	// process light
	if( light.position.w == 0. )
		DirectionalLight( light, FrontMaterial, ecPosition3, normal,
							ambient, diffuse, specular );
						
	else if( light.spotCutoff == 180.0 )
		PointLight( light, FrontMaterial, eye, ecPosition3, normal,
					ambient, diffuse, specular );
					
	else SpotLight( light, FrontMaterial, eye, ecPosition3, normal,
					ambient, diffuse, specular );
	
	if( shadeFactor != 1. ) {
		specular = vec3( 0., 0., 0. );
		diffuse = diffuse.rgb * shadeFactor;
	}
}

void processLights( in vec3 ecPosition3, in vec3 normal,
			inout vec3 ambient, inout vec3 diffuse, inout vec3 specular )
{
	vec3 eye = vec3( 0., 0., 1. ); // -normalize(ex_ecPosition3)

	for( int i=0; i < NumEnabledLights; i++ ) {
		if( LightSource[ i ].position.w == 0. )
			DirectionalLight( LightSource[ i ], FrontMaterial,
							ecPosition3, normal, ambient, diffuse, specular );
		else if( LightSource[ i ].spotCutoff == 180.0 )
			PointLight( LightSource[ i ], FrontMaterial,
						eye, ecPosition3, normal,
						ambient, diffuse, specular );
		else SpotLight( LightSource[ i ], FrontMaterial,
						eye, ecPosition3, normal,
						ambient, diffuse, specular );
	}
}

// **************** Lighting with Shadow Maps ****************

bool isShadowed( in sampler2D shadowMap, in vec3 coord, in vec2 offset ) {
	//const float pixelOffset = 1./3200.;
	//const float pixelOffset = 1./6400.;
	//const float pixelOffset = 1./12800.;
	const float pixelOffset = 1./9600.;
	//const float epsilon = 0.00001;
	const float epsilon = 0.;

	float depth = epsilon + texture2D( shadowMap, coord.st + offset * pixelOffset ).z;

	if( depth >= 1. ) return false;
	return ( (depth < coord.z) ? true : false );
}

// returns 1. if not shaded, 0. if total in shadow
float getShadeFactorAA( in sampler2D shadowMap, in vec3 coord ) {
	float total = 1.;

	if( (coord.s >= 0.) && (coord.s <= 1.) &&
		(coord.t >= 0.) && (coord.t <= 1.) )
	{
		const float factor = -.25;
			
		// simple 4 sample anti aliasing
		total += ( isShadowed( shadowMap, coord, vec2(-0.5, -0.5) ) ? factor : 0. );
		total += ( isShadowed( shadowMap, coord, vec2( 0.5, -0.5) ) ? factor : 0. );
		total += ( isShadowed( shadowMap, coord, vec2(-0.5,  0.5) ) ? factor : 0. );
		total += ( isShadowed( shadowMap, coord, vec2( 0.5,  0.5) ) ? factor : 0. );	
	}
	return total;
	
}
// returns 1. if not shaded, 0. if total in shadow
float getShadeFactorD( in sampler2D shadowMap, in vec3 coord ) {
	float total = 1.;

	if( (coord.s >= 0.) && (coord.s <= 1.) &&
		(coord.t >= 0.) && (coord.t <= 1.) )
	{
		const float factor = -.25;
		
		// antialiasing kine to dithering
		vec3 o = coord + vec3( mod( floor( gl_FragCoord.xy ), 2.0 ), 0. );
		total += ( isShadowed( shadowMap, o, vec2(-1.5,  1.5) ) ? factor : 0. );
		total += ( isShadowed( shadowMap, o, vec2( 0.5,  1.5) ) ? factor : 0. );
		total += ( isShadowed( shadowMap, o, vec2(-1.5, -0.5) ) ? factor : 0. );
		total += ( isShadowed( shadowMap, o, vec2( 0.5, -0.5) ) ? factor : 0. );	
	}
	return total; 
}