#ifndef NX_PHYSICS_NX_MESHDATA
#define NX_PHYSICS_NX_MESHDATA
/** \addtogroup cloth
  @{
*/
/*----------------------------------------------------------------------------*\
|
|						Public Interface to Ageia PhysX Technology
|
|							     www.ageia.com
|
\*----------------------------------------------------------------------------*/

/**
\brief Very similar to #NxMeshFlags used for the #NxSimpleTriangleMesh type.
*/
enum NxMeshDataFlags
	{
	/**
	\brief Denotes the use of 16-bit vertex indices.
	*/
	NX_MDF_16_BIT_INDICES				=	1 << 0, 
	};

/**
\brief Enum with flag values to be used in NxMeshData::dirtyBufferFlagsPtr.
*/
enum NxMeshDataDirtyBufferFlags
	{
	/**
	\brief Denotes a change in the vertex position buffer.
	*/
	NX_MDF_VERTICES_POS_DIRTY			=	1 << 0, 
	/**
	\brief Denotes a change in the vertex normal buffer.
	*/
	NX_MDF_VERTICES_NORMAL_DIRTY		=	1 << 1, 
	/**
	\brief Denotes a change in the index buffer.
	*/
	NX_MDF_INDICES_DIRTY				=	1 << 2, 
	/**
	\brief Denotes a change in the parent index buffer.
	*/
	NX_MDF_PARENT_INDICES_DIRTY			=	1 << 3, 
	};

/**
\brief Descriptor-like user-side class for describing mesh data.

This data type is used for specifying how the SDK is supposed to pass generated mesh data. This is used 
to pass simulated cloth meshes back to the user.

This class is very similar to NxSimpleTriangleMesh, with the difference that this user buffer 
wrapper is used to let the SDK write to user buffers instead of reading from them.
*/
class NxMeshData
	{
	public:

	/**
	\brief 	The pointer to the user specified buffer for vertex positions.

	A vertex position consists of three consecutive 32 bit floats.
	If the pointer is not initialized (NULL), no data is returned.
	*/
	void*					verticesPosBegin;

	/**
	\brief 	The pointer to the user specified buffer for vertex normals. 

	A vertex normal consists of three consecutive 32 bit floats.
	If the pointer is not initialized (NULL), no data is returned.
	*/
	void*					verticesNormalBegin;

	/**
	\brief Specifies the distance of two vertex position start addresses in bytes.
	*/
	NxI32					verticesPosByteStride;

	/**
	\brief Specifies the distance of two vertex normal start addresses in bytes.
	*/
	NxI32					verticesNormalByteStride;

	/**
	\brief The maximal number of vertices which can be stored in the user vertex buffers.
	*/
	NxU32					maxVertices;

	/**
	\brief Must point to the user allocated memory holding the number of vertices stored in the user vertex 
	buffers.
	
	If the SDK writes to a given vertex buffer, it also sets the numbers of elements written.
	*/
	NxU32*					numVerticesPtr;

	/**
	\brief The pointer to the user specified buffer for vertex indices.
	
	An index consist of one 32 or 16 bit integers, depending on whether NX_MDF_16_BIT_INDICES has been set.
	
	If the pointer is not initialized (NULL), no data is returned.
	*/
	void*					indicesBegin;

	/**
	\brief Specifies the distance of two vertex indices start addresses in bytes.
	*/
	NxI32					indicesByteStride;

	/**
	\brief The maximal number of indices which can be stored in the user index buffer.
	*/
	NxU32					maxIndices;

	/**
	\brief Must point to the user allocated memory holding the number of vertex triplets used to define 
	triangles.
	
	If the SDK writes to a given triangle index buffer, it also sets the number of
	triangles written.
	*/
	NxU32*					numIndicesPtr;

	/**
	\brief The pointer to the user specified buffer for vertex parent indices. 
	
	An index consist of one 32 or 16 bit integers, depending on whether NX_MDF_16_BIT_INDICES has been set.

	Parent indices are provided when vertices are duplicated by the SDK (e.g. cloth tearing).
	The parent index of an original vertex is its position in the verticesPos buffer. The
	parent index of a vertex generated by duplication is the index of the vertex it was copied from.
	
	If the pointer is not initialized (NULL), no data is returned.
	*/
	void*					parentIndicesBegin;

	/**
	\brief Specifies the distance of two vertex parent indices start addresses in bytes.
	*/
	NxI32					parentIndicesByteStride;

	/**
	\brief The maximal number of parent indices which can be stored in the user parent index buffer.
	*/
	NxU32					maxParentIndices;

	/**
	\brief Must point to the user allocated memory holding the number of vertex parent indices
	
	If the SDK writes to a given vertex parent index buffer, it also sets the number of
	parent indices written. 
	*/
	NxU32*					numParentIndicesPtr;

	/**
	\brief Must point to the user allocated memory holding the dirty buffer flags

	If the SDK changes the content of a given buffer, it also sets the corresponding flag of
	type #NxMeshDataDirtyBufferFlags. This functionality is only supported in conjunction with
	cloth yet. The returned value for other features is undefined.
	*/
	NxU32*					dirtyBufferFlagsPtr;

	/**
	\brief Flags of type #NxMeshDataFlags
	*/
	NxU32					flags;

	const char*				name;			//!< Possible debug name. The string is not copied by the SDK, only the pointer is stored.

	NX_INLINE ~NxMeshData();

	/**
	\brief (Re)sets the structure to the default.	
	*/
	NX_INLINE void setToDefault();

	/**
	\brief Returns true if the current settings are valid
	*/
	NX_INLINE bool isValid() const;

	/**
	\brief Constructor sets to default.
	*/
	NX_INLINE	NxMeshData();
	};

NX_INLINE NxMeshData::NxMeshData()
	{
	setToDefault();
	}

NX_INLINE NxMeshData::~NxMeshData()
	{
	}

NX_INLINE void NxMeshData::setToDefault()
	{
	verticesPosBegin				= NULL;
	verticesNormalBegin				= NULL;
	verticesPosByteStride			= 0;
	verticesNormalByteStride		= 0;
	maxVertices						= 0;
	numVerticesPtr					= NULL;
	indicesBegin					= NULL;
	indicesByteStride				= 0;
	maxIndices						= 0;
	numIndicesPtr					= NULL;
	parentIndicesBegin				= NULL;
	parentIndicesByteStride			= 0;
	maxParentIndices				= 0;
	numParentIndicesPtr				= NULL;
	dirtyBufferFlagsPtr				= NULL;
	flags							= 0;
	name							= NULL;
	}

NX_INLINE bool NxMeshData::isValid() const
	{
	if (numVerticesPtr && !(verticesPosBegin || verticesNormalBegin)) return false;
	if (!numVerticesPtr && (verticesPosBegin || verticesNormalBegin)) return false;

	if (!numIndicesPtr && indicesBegin) return false;
	if (numIndicesPtr && !indicesBegin) return false;

	if (!numParentIndicesPtr && parentIndicesBegin) return false;
	if (numParentIndicesPtr && !parentIndicesBegin) return false;

	if (verticesPosBegin && !verticesPosByteStride) return false;
	if (verticesNormalBegin && !verticesNormalByteStride) return false;
	if (indicesBegin && !indicesByteStride) return false;
	if (parentIndicesBegin && !parentIndicesByteStride) return false;
				
	return true;
	}

/** @} */
#endif

//AGCOPYRIGHTBEGIN
///////////////////////////////////////////////////////////////////////////
// Copyright (c) 2005 AGEIA Technologies.
// All rights reserved. www.ageia.com
///////////////////////////////////////////////////////////////////////////
//AGCOPYRIGHTEND
