// Specular GLSL Vertex Shader
// Written by Florian Rudolf (corrail@gmx.at)
// Copyright (c) 2004; All right reserved

uniform float Time;
const int max_waves = 3;
uniform vec2 WaveDirection[max_waves];
uniform float WaveFrequency[max_waves];
uniform float WaveHeight[max_waves];
uniform float WavePhaseSpeed[max_waves];

attribute vec2 a2v_Texcoord;

varying vec3 v2f_Normal;
varying vec3 v2f_LightVec;
varying vec3 v2f_ViewVec;
varying vec2 v2f_Texcoord;

float getWaveHeight(vec2 _position, float _time)
{
	float height = 0.0;
	for (int i = 0; i < max_waves; i++)
	{
		height += WaveHeight[i] * sin(dot(WaveDirection[i], _position)
			* WaveFrequency[i] + _time * WavePhaseSpeed[i]);
	}
	return height;
}

vec3 getWaveNormal(vec2 _position, float _time)
{
	vec3 Normal = vec3(0.0, 0.0, 1.0);

	for (int i = 0; i < max_waves; i++)
	{
		Normal.x -= WaveHeight[i] * WaveDirection[i].x *
			WaveFrequency[i] * cos(dot(WaveDirection[i], _position)
			* WaveFrequency[i] + _time * WavePhaseSpeed[i]);

		Normal.y -= WaveHeight[i] * WaveDirection[i].y *
			WaveFrequency[i] * cos(dot(WaveDirection[i], _position)
			* WaveFrequency[i] + _time * WavePhaseSpeed[i]);
	}

	return normalize(Normal);
}

void main()
{
	// Vertex transformation
	vec4 Vertex = gl_Vertex;
	Vertex.z += getWaveHeight(vec2(gl_Vertex), Time);
	gl_Position = gl_ModelViewProjectionMatrix * Vertex;

	v2f_Normal = gl_NormalMatrix * getWaveNormal(vec2(gl_Vertex), Time);
	
	// Calculating Light vector in ModelView Space
	vec3 Vertex_Model = vec3(gl_ModelViewMatrix * gl_Vertex);	
	vec3 LightVec = vec3(gl_LightSource[0].position) - Vertex_Model;

	v2f_LightVec = LightVec;
	v2f_ViewVec = -Vertex_Model;
	
	// Passing texture coordinates to fragment shader
	v2f_Texcoord = a2v_Texcoord;
}