#include "SkyBox.hpp"
#include <iostream>
#include <sstream>
#include <glm/glm.hpp>
#include <glm\gtc\matrix_transform.hpp>
#include <glm\gtc\type_ptr.hpp>
using namespace cgue::scene;

SkyBox::SkyBox(cgue::util::Shader* _shader)
:shader(_shader){

	//shader = _shader;
	vao = 0;
	vbo = 0;

	glGenVertexArrays(1, &vao);
	glGenBuffers(1, &vbo);
	glBindVertexArray(vao);
	glBindBuffer(GL_ARRAY_BUFFER, vbo);
	glBufferData(GL_ARRAY_BUFFER, sizeof(positions), &positions, GL_STATIC_DRAW);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 3 * sizeof(GLfloat), (GLvoid*)0);
	glBindVertexArray(0);

	// Cubemap (Skybox)
	std::vector<std::string> faces;

	faces.push_back("Models//cubemap//clouds//clouds_lf.tga");
	faces.push_back("Models//cubemap//clouds//clouds_rt.tga");
	faces.push_back("Models//cubemap//clouds//clouds_up.tga");
	faces.push_back("Models//cubemap//clouds//clouds_dn.tga");
	faces.push_back("Models//cubemap//clouds//clouds_ft.tga");
	faces.push_back("Models//cubemap//clouds//clouds_bk.tga");
	
	//faces.push_back("Models//cubemap//mp_organic//organic_lf.tga");
	//faces.push_back("Models//cubemap//mp_organic//organic_rt.tga");
	//faces.push_back("Models//cubemap//mp_organic//organic_up.tga");
	//faces.push_back("Models//cubemap//mp_organic//organic_dn.tga");
	//faces.push_back("Models//cubemap//mp_organic//organic_bk.tga");
	//faces.push_back("Models//cubemap//mp_organic//organic_ft.tga");
	
	cubemapTexture = loadCubemap(faces);
};
SkyBox::~SkyBox() {
	glDeleteBuffers(1, &vbo);
	glDeleteVertexArrays(1, &vao);
	glDeleteTextures(1, &cubemapTexture);
}

GLuint SkyBox::loadCubemap(std::vector<std::string> faces) {
	GLuint textureID;
	glGenTextures(1, &textureID);

	int width, height;
	FIBITMAP *bitmap, *pImage;

	glBindTexture(GL_TEXTURE_CUBE_MAP, textureID);
	GLint i = 0;
	for (auto &face : faces) {
		bitmap = FreeImage_Load(FreeImage_GetFileType(face.c_str(), 0), face.c_str());
		FreeImage_FlipVertical(bitmap);
		pImage = FreeImage_ConvertTo32Bits(bitmap);

		FreeImage_Unload(bitmap);

		width = FreeImage_GetWidth(pImage);
		height = FreeImage_GetHeight(pImage);

		glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA8, width, height, 0, GL_BGRA, GL_UNSIGNED_BYTE, (void*)FreeImage_GetBits(pImage));

		FreeImage_Unload(pImage);
		i++;
	}
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glGenerateMipmap(GL_TEXTURE_CUBE_MAP);
	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);

	return textureID;
};

void SkyBox::draw() {
	glDepthFunc(GL_LEQUAL);

	shader->useShader();

	auto model_location = glGetUniformLocation(shader->programHandle, "model");
	glUniformMatrix4fv(model_location, 1, GL_FALSE, glm::value_ptr(model));

	glBindVertexArray(vao);
	glActiveTexture(GL_TEXTURE0);
	glUniform1i(glGetUniformLocation(shader->programHandle, "skybox"), 0);
	glBindTexture(GL_TEXTURE_CUBE_MAP, cubemapTexture);
	glDrawArrays(GL_TRIANGLES, 0, 36);
	glBindVertexArray(0);

	glDepthFunc(GL_LESS);
}

void SkyBox::bindToTexture() {
	glActiveTexture(GL_TEXTURE15);
	glBindTexture(GL_TEXTURE_CUBE_MAP, cubemapTexture);
}

GLuint SkyBox::getCubemapTexture() {
	return cubemapTexture;
}

GLuint SkyBox::getVAO() {
	return vao;
}

const float SkyBox::positions[] = {
	-1.0f, 1.0f, -1.0f,
	-1.0f, -1.0f, -1.0f,
	1.0f, -1.0f, -1.0f,
	1.0f, -1.0f, -1.0f,
	1.0f, 1.0f, -1.0f,
	-1.0f, 1.0f, -1.0f,

	-1.0f, -1.0f, 1.0f,
	-1.0f, -1.0f, -1.0f,
	-1.0f, 1.0f, -1.0f,
	-1.0f, 1.0f, -1.0f,
	-1.0f, 1.0f, 1.0f,
	-1.0f, -1.0f, 1.0f,

	1.0f, -1.0f, -1.0f,
	1.0f, -1.0f, 1.0f,
	1.0f, 1.0f, 1.0f,
	1.0f, 1.0f, 1.0f,
	1.0f, 1.0f, -1.0f,
	1.0f, -1.0f, -1.0f,

	-1.0f, -1.0f, 1.0f,
	-1.0f, 1.0f, 1.0f,
	1.0f, 1.0f, 1.0f,
	1.0f, 1.0f, 1.0f,
	1.0f, -1.0f, 1.0f,
	-1.0f, -1.0f, 1.0f,

	-1.0f, 1.0f, -1.0f,
	1.0f, 1.0f, -1.0f,
	1.0f, 1.0f, 1.0f,
	1.0f, 1.0f, 1.0f,
	-1.0f, 1.0f, 1.0f,
	-1.0f, 1.0f, -1.0f,

	-1.0f, -1.0f, -1.0f,
	-1.0f, -1.0f, 1.0f,
	1.0f, -1.0f, -1.0f,
	1.0f, -1.0f, -1.0f,
	-1.0f, -1.0f, 1.0f,
	1.0f, -1.0f, 1.0f
};


void SkyBox::setSamplingQuality(int minQuality, int magQuality){
	glBindTexture(GL_TEXTURE_CUBE_MAP, cubemapTexture);

	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, minQuality);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, magQuality);
}