#include "SceneObject.hpp"

#include <glm/glm.hpp>
#include <glm\gtc\matrix_transform.hpp>
#include <iostream>
#include <glm/gtx/string_cast.hpp>

using namespace cgue::scene;

SceneObject::SceneObject(glm::mat4& _modelMatrix, ObjectType _objectType)
:modelMatrix(_modelMatrix), objectType(_objectType) {
	rotationMatrix = glm::mat4(1.0f);
	scaleFactor = 1.0f;
	radius = 0.5f; // default radius
	height = 2.0f; //default height
	glowing = false;
}

SceneObject::~SceneObject() {

}

void SceneObject::rotate(cgue::scene::Direction dir, const float& degree) {
	float angle = glm::radians(degree * 1.0f);

	glm::vec3 rotationVector = glm::vec3(0.0f);

	if (dir == LEFT || dir == RIGHT) {
		rotationVector = glm::vec3(0, 1, 0);
	}
	else {
		rotationVector = glm::vec3(1, 0, 0);
	}

	modelMatrix = glm::rotate(modelMatrix, angle, rotationVector);
	rotationMatrix = glm::rotate(rotationMatrix, angle, rotationVector);
}

void SceneObject::translate(glm::highp_vec3 v){
	modelMatrix = glm::translate(modelMatrix, v);
}

void SceneObject::scale(glm::highp_vec3 v) {
	modelMatrix = glm::scale(modelMatrix, v);
	scaleFactor *= v.x;
}

glm::vec3 SceneObject::position() {
	return glm::vec3(modelMatrix[3].x, modelMatrix[3].y, modelMatrix[3].z);
}

bool SceneObject::detectHit(SceneObject* other) {
	auto myPos = position();
	auto otherPos = other->position();

	auto vec = otherPos - myPos;
	float distance = sqrt(vec.x*vec.x + vec.y*vec.y + vec.z*vec.z);
	
	if (distance < (radius + other->radius)) {
		return true;
	}
	return false;
}

bool SceneObject::detectFrontHit(SceneObject* other) {
	auto myPos = position();
	auto otherPos = other->position();

	// front plane of hero
	glm::vec3 planeCenter(myPos.x, myPos.y, myPos.z + radius);
	glm::vec3 planeNormal(0.0f, 0.0f, 1.0f);

	auto vec = otherPos - planeCenter;
	float distance = glm::dot(vec, planeNormal);

	// other scene object is behind front plane 
	// && above or in same hight of current object
	if (myPos.y+height < other->height
		&& distance < 0.0) {
		return true;
	}
	return false;
}