/*reference: http://www.learnopengl.com/#!Model-Loading/Model */

#include "Model.hpp"
#include "Mesh.hpp"

using namespace cgue;
using namespace cgue::scene;
using namespace cgue::scene::mesh;

Model::Model(glm::mat4& _modelMatrix, ObjectType _objectType, util::Shader* _shader, GLchar* path)
: SceneObject(_modelMatrix, _objectType), shader(_shader), light(nullptr) {

	normalMapping = true;

	initModel(path);
	//radius = 0.7f; // @TODO calculate radius somehow
}

Model::~Model() {
	for (auto meshElem : meshes) {
		delete meshElem; meshElem = nullptr;
	}
}


void Model::draw(){

	shader->useShader();

	auto model_location = glGetUniformLocation(shader->programHandle, "model");
	glUniformMatrix4fv(model_location, 1, GL_FALSE, glm::value_ptr(modelMatrix));

	auto t_i_model = glm::transpose(glm::inverse(modelMatrix));
	auto t_i_model_location = glGetUniformLocation(shader->programHandle, "t_i_model");
	glUniformMatrix4fv(t_i_model_location, 1, GL_FALSE, glm::value_ptr(t_i_model));

	// booleans for shader
	/*GLboolean blinn = true;
	auto blinn_location = glGetUniformLocation(shader->programHandle, "blinnSwitch");
	glUniform1i(blinn_location, blinn);*/

	auto normalMapping_location = glGetUniformLocation(shader->programHandle, "normalMapping");
	glUniform1i(normalMapping_location, normalMapping);

	//lamp light properties
	GLint lightPos_location = glGetUniformLocation(shader->programHandle, "light.position");
	GLint lightAmbient_location = glGetUniformLocation(shader->programHandle, "light.ambient");
	GLint lightDiffuse_location = glGetUniformLocation(shader->programHandle, "light.diffuse");
	GLint lightSpecular_location = glGetUniformLocation(shader->programHandle, "light.specular");

	//check if lighting is available in shader //TODO better check
	if (light == nullptr) {
		lightPos = glm::vec3(1.0f, 10.0f, 0.0f);
		lightColor = glm::vec3(0.0f, 0.0f, 1.0f);
	}
	else {
		lightPos = light->getLightPos();
		lightColor = light->getLightColor();
	}
	//lamp light porperties
	//diffuse and ambient light should only mildly influence the object color
	glm::vec3 ambientLight = lightColor * glm::vec3(0.2f);
	glm::vec3 diffuseLight = lightColor * glm::vec3(0.5f);
	glm::vec3 specularLight = glm::vec3(1.0f);

	glUniform3f(lightAmbient_location, ambientLight.x, ambientLight.y, ambientLight.z);
	glUniform3f(lightDiffuse_location, diffuseLight.x, diffuseLight.y, diffuseLight.z);
	glUniform3f(lightSpecular_location, specularLight.x, specularLight.y, specularLight.z);
	glUniform3f(lightPos_location, lightPos.x, lightPos.y, lightPos.z);	//set light Position ; position of lamp

	glUniform1i(glGetUniformLocation(shader->programHandle, "skybox"), 15);

	for (auto &meshElem : meshes) {
		meshElem->draw();
	}

	
}


void Model::drawCustomShader(util::Shader* customshader) {
	util::Shader* tmp = shader;
	shader = customshader;
	for (auto &meshElem : meshes) {
		meshElem->setShader(shader);
	}
	draw();
	shader = tmp;
	for (auto &meshElem : meshes) {
		meshElem->setShader(shader);
	}

}
void Model::update(float time_delta) {
	if (objectType == ObjectType::GOOD 
		|| objectType == ObjectType::BAD 
		|| objectType == ObjectType::BONUS1
		|| objectType == ObjectType::BONUS2) {
		rotate(Direction::LEFT, 45.0f * time_delta);
		//modelMatrix = glm::rotate(modelMatrix, glm::radians(45.0f) * time_delta, glm::vec3(0, 1, 0));	//45degree(convert to rad!!) per second, glm::Vec3() axis which get rotated
	}
}

void Model::initModel(const std::string &path) {

	Assimp::Importer import;
	//aiProcess_Triangulate: force all shapes of model to triangles
	//aiProcess_FlipUVs: flip UV coordiantes of textures
	const aiScene* scene = import.ReadFile(path, 
		aiProcess_CalcTangentSpace |
		aiProcess_Triangulate |
		aiProcess_JoinIdenticalVertices |
		aiProcess_SortByPType); //| aiProcess_FlipUVs);

	if (!scene || scene->mFlags == AI_SCENE_FLAGS_INCOMPLETE || !scene->mRootNode) {
		std::cout << "Error AASIMP" << import.GetErrorString() << std::endl;
		system("PAUSE");
		exit(EXIT_FAILURE);
	}

	directory = path.substr(0, path.find_last_of('/')); // ERROR!!!
	processNode(scene->mRootNode, scene);
}

//recursive process all aiNodes
void Model::processNode(aiNode* node, const aiScene* scene)
{
	// Process all the node's meshes (if any)
	for (GLuint i = 0; i < node->mNumMeshes; i++) {

		aiMesh* mesh = scene->mMeshes[node->mMeshes[i]];
		this->meshes.push_back(this->processMesh(mesh, scene));
	}
	// Then do the same for each of its children
	for (GLuint i = 0; i < node->mNumChildren; i++) {

		this->processNode(node->mChildren[i], scene);
	}
}

mesh::Mesh* Model::processMesh(aiMesh* mesh, const aiScene* scene) {
	std::vector<Vertex> vertices;
	std::vector<GLuint> indices;
	std::vector<Texture> textures;

	//process vertices - fill struct Vertex in Mesh.hpp
	for (GLuint i = 0; i < mesh->mNumVertices; i++) {

		//load position, normals, textures and tangents of aiMesh
		Vertex vertex;
		glm::vec3 vector(mesh->mVertices[i].x, mesh->mVertices[i].y, mesh->mVertices[i].z);
		vertex.Position = vector;

		vector.x = mesh->mNormals[i].x;
		vector.y = mesh->mNormals[i].y;
		vector.z = mesh->mNormals[i].z;
		vertex.Normal = vector;

		if (mesh->mTextureCoords[0]) {	// Does the mesh contain texture coordinates?

			glm::vec2 vec;
			//always take the first set of texture coordinates
			vec.x = mesh->mTextureCoords[0][i].x;
			vec.y = mesh->mTextureCoords[0][i].y;
			vertex.TexCoords = vec;

			// tangent only available with texture coordinates
			glm::vec3 tangent;
			tangent.x = mesh->mTangents[i].x;
			tangent.y = mesh->mTangents[i].y;
			tangent.z = mesh->mTangents[i].z;
			vertex.Tangent = tangent;

		}
		else {
			vertex.TexCoords = glm::vec2(0.0f, 0.0f);
			vertex.Tangent = glm::vec3(1.0f);
		}

		vertices.push_back(vertex);
	}



	//process faces (defines the vertices which need to be drawn)
	for (GLuint i = 0; i < mesh->mNumFaces; i++)
	{
		aiFace face = mesh->mFaces[i];
		for (GLuint j = 0; j < face.mNumIndices; j++)
			indices.push_back(face.mIndices[j]);
	}

	// process materials (diffuse/specular/normal maps)
	if (mesh->mMaterialIndex >= 0)	{

		aiMaterial* material = scene->mMaterials[mesh->mMaterialIndex];

		std::vector<Texture> diffuseMaps = this->loadMaterialTextures(material, aiTextureType_DIFFUSE, "texture_diffuse");
		textures.insert(textures.end(), diffuseMaps.begin(), diffuseMaps.end());
		
		std::vector<Texture> specularMaps = this->loadMaterialTextures(material, aiTextureType_SPECULAR, "texture_specular");
		textures.insert(textures.end(), specularMaps.begin(), specularMaps.end());
		
		std::vector<Texture> normalMaps = this->loadMaterialTextures(material, aiTextureType_HEIGHT, "texture_normal"); // aiTextureType_NORMALS does not work for .obj!!!
		textures.insert(textures.end(), normalMaps.begin(), normalMaps.end());
	}

	return new Mesh(shader, vertices, indices, textures);

}

//load texture
std::vector<mesh::Texture> Model::loadMaterialTextures(aiMaterial* mat, aiTextureType type, std::string typeName)	{

	std::vector<Texture> textures;

	for (GLuint i = 0; i < mat->GetTextureCount(type); i++) {

		aiString str;
		mat->GetTexture(type, i, &str);
		GLboolean skip = false;

		//if a texture gets used again, it does not need to be imported again
		for (GLuint i = 0; i < textures_loaded.size(); i++) {
			if (str == textures_loaded[i].path) {
				textures.push_back(textures_loaded[i]);
				skip = true;
				break;
			}

		}
		//only load texture if it is new
		if (!skip) {
			Texture texture;
			texture.id = TextureFromFile(str.C_Str(), this->directory);
			texture.type = typeName;
			texture.path = str;
			textures.push_back(texture);
			textures_loaded.push_back(texture);
		}
	}

	return textures;

}


GLint Model::TextureFromFile(const char* path, std::string directory) {

	//Generate texture ID and load texture data 
	std::string filename = std::string(path);
	filename = directory + '/' + filename;

	int width, height;
	FIBITMAP* bitmap = FreeImage_Load(FreeImage_GetFileType(filename.c_str(), 0), filename.c_str());


	FIBITMAP *pImage = FreeImage_ConvertTo32Bits(bitmap);
	FreeImage_Unload(bitmap);

	width = FreeImage_GetWidth(pImage);
	height = FreeImage_GetHeight(pImage);

	GLuint textureID;
	glGenTextures(1, &textureID);
	// Assign texture to ID
	glBindTexture(GL_TEXTURE_2D, textureID);
	//duck colors are correct in irfanview when rgb->bgr 
	//for duck: use second color scheme GL_BGRA leave swizle parameter in shader to .rgb or use GL_BRGB and change swizzle to .bgr
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, (void*)FreeImage_GetBits(pImage));
	//glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, width, height, 0, GL_BGRA, GL_UNSIGNED_BYTE, (void*)FreeImage_GetBits(pImage));
	glGenerateMipmap(GL_TEXTURE_2D);

	// Parameters
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	//glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	//glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	//unbind

	glBindTexture(GL_TEXTURE_2D, 0);
	FreeImage_Unload(pImage);
	return textureID;
}

void Model::setLight(Light* _light){
	light = _light;
}

void Model::setSamplingQuality(int minQuality, int magQuality){
	for (GLuint i = 0; i < textures_loaded.size(); i++) {
		glBindTexture(GL_TEXTURE_2D, textures_loaded[i].id);

		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, minQuality);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, magQuality);
	}
}

void Model::changeNormalMapping() {
	if (normalMapping) {
		normalMapping = false;
	}
	else {
		normalMapping = true;
	}
}