//reference: http://learnopengl.com/#!Advanced-OpenGL/Framebuffers
#include "FrameBufferQuad.hpp"


using namespace cgue;
using namespace cgue::scene;


FrameBufferQuad::FrameBufferQuad(util::Shader* _shader, int _screenWidth, int _sreenHeight)
:shader(_shader), screenWidth(_screenWidth), screenHeight(_sreenHeight) {

	init();
}

FrameBufferQuad::~FrameBufferQuad() {
	glDeleteBuffers(1, &vbo);
	glDeleteFramebuffers(1, &fbo);
	glDeleteRenderbuffers(1, &rbo);
	glDeleteVertexArrays(1, &vao);
	glDeleteTextures(1, &texColBuffer);
}

void FrameBufferQuad::init(){

	//quad (the whole screen)
	glGenVertexArrays(1, &vao);
	glGenBuffers(1, &vbo);
	glBindVertexArray(vao);
	glBindBuffer(GL_ARRAY_BUFFER, vbo);
	glBufferData(GL_ARRAY_BUFFER, sizeof(quadVertices), &quadVertices, GL_STATIC_DRAW);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 4 * sizeof(GLfloat), (GLvoid*)0);
	glEnableVertexAttribArray(1);
	glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 4 * sizeof(GLfloat), (GLvoid*)(2 * sizeof(GLfloat)));
	glBindVertexArray(0);

	//framebuffer
	glGenFramebuffers(1, &fbo);
	glBindFramebuffer(GL_FRAMEBUFFER, fbo);

	//texture of frame buffer
	glGenTextures(1, &texColBuffer);
	glBindTexture(GL_TEXTURE_2D, texColBuffer);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, screenWidth, screenHeight, 0, GL_RGB, GL_UNSIGNED_BYTE, NULL);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glBindTexture(GL_TEXTURE_2D, 0);

	// render buffer (depth buffer)
	glGenRenderbuffers(1, &rbo);
	glBindRenderbuffer(GL_RENDERBUFFER, rbo);
	glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT, screenWidth, screenHeight);
	//glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH24_STENCIL8, screenWidth, screenHeight);
	glBindRenderbuffer(GL_RENDERBUFFER, 0);
	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, rbo);
	//glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_STENCIL_ATTACHMENT, GL_RENDERBUFFER, rbo);


	// Set "texColBuffer" as color attachment #0
	glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, texColBuffer, 0);


	if (glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE){
		std::cout << "ERROR::FRAMEBUFFER:: Framebuffer is not complete!" << std::endl;
		system("PAUSE");
		exit(EXIT_FAILURE);

	}

	glBindFramebuffer(GL_FRAMEBUFFER, 0);
	//glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);	//if everything is writen to our framebuffer this should just display an empty quad

}



void FrameBufferQuad::bind(){
	glBindFramebuffer(GL_FRAMEBUFFER, fbo);
	// Clear all attached buffers        
	glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

void FrameBufferQuad::unbind() {
	//bind to standard framebuffer again
	glBindFramebuffer(GL_FRAMEBUFFER, 0);
	// Clear all relevant buffers
	//glClearColor(0.0f, 0.0f, 0.0f, 1.0f); // Set clear color to white (not really necessery actually, since we won't be able to see behind the quad anyways)
	glClear(GL_COLOR_BUFFER_BIT);
}




void FrameBufferQuad::draw() {
	glDisable(GL_DEPTH_TEST); // We don't care about depth information when rendering a single quad
	
	// Draw Screen
	shader->useShader();

	auto widthLocation = glGetUniformLocation(shader->programHandle, "width");
	glUniform1f(widthLocation, (float)screenWidth);

	auto heightLocation = glGetUniformLocation(shader->programHandle, "height");
	glUniform1f(heightLocation, (float)screenHeight);

	glBindVertexArray(vao);
	glBindTexture(GL_TEXTURE_2D, texColBuffer);	// Use the color attachment texture as the texture of the quad plane
	glDrawArrays(GL_TRIANGLES, 0, 6);
	glBindVertexArray(0);
	
	glEnable(GL_DEPTH_TEST);
}

void FrameBufferQuad::unbindAndDraw() {
	unbind();
	draw();
}

void FrameBufferQuad::activateTexture() {
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, texColBuffer);
}

void FrameBufferQuad::deactivateTexture() {
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, 0);
}

const GLfloat FrameBufferQuad::quadVertices[QUAD_VERTEX_COUNT] = {   // Vertex attributes for a quad that fills the entire screen in Normalized Device Coordinates.
	// Positions   // TexCoords
	-1.0f, 1.0f, 0.0f, 1.0f,
	-1.0f, -1.0f, 0.0f, 0.0f,
	1.0f, -1.0f, 1.0f, 0.0f,

	-1.0f, 1.0f, 0.0f, 1.0f,
	1.0f, -1.0f, 1.0f, 0.0f,
	1.0f, 1.0f, 1.0f, 1.0f
};