#include "LoopHelper.hpp"
#include "glm/gtx/string_cast.hpp"


using namespace cgue::loop;
using namespace cgue::scene;
using namespace cgue::util;
using namespace cgue::loader;

LoopHelper::LoopHelper(GLFWwindow* _window, SoundHandler* _soundhandler)
:window(_window) {

	shaderLoader = new ShaderLoader();
	skybox = new SkyBox(shaderLoader->get("SkyBox"));
	sceneObjectLoader = new SceneObjectLoader(shaderLoader, skybox);
	textureLoader = new TextureLoader();
	camera = new Camera(sceneObjectLoader->getHeroParts()->heroBody);
	camera->toggleHeroAttachment();
	frustum = new Frustum();


	// set viewport
	glfwGetFramebufferSize(window, &width, &height);
	this->initializeProjection();
	glViewport(0, 0, width, height);


	// initialize frame buffer
	framebufferquad = new FrameBufferQuad(shaderLoader->get("FrameBufferScreen"), width, height);
	depthbufferquad = new ShadowMapBuffer(shaderLoader->get("DepthBufferDebug"), width, height);
	objectBuffer = new FrameBufferQuad(shaderLoader->get("FrameBufferScreen"), width, height);
	horizontalGlowBuffer = new FrameBufferQuad(shaderLoader->get("HorizontalGlowBufferScreen"), width, height);
	verticalGlowBuffer = new FrameBufferQuad(shaderLoader->get("VerticalGlowBufferScreen"), width, height);

	
	text = new Text(shaderLoader->get("Font"), width, height);
	gameLogic = new GameLogic(sceneObjectLoader, text, _soundhandler);
	toast = new Toast(text);
	helpView = new HelpView(text);

	displayFrameTime = true;
	displayWireFrame = false;
	textureSamplingQuality = TextureSamplingQuality::TSQ_BILINEAR;
	mipMapQuality = MipMapQuality::MMQ_Linear;
	viewFrustumCulling = true;
	transparency = true;
	displayHelp = false;
	
	gameStarted = false;
	gamePause = false;

	fps = 0;
}

LoopHelper::~LoopHelper()
{
	delete shaderLoader; shaderLoader = nullptr;
	delete sceneObjectLoader; sceneObjectLoader = nullptr;
	delete camera; camera = nullptr;
	delete skybox; skybox = nullptr;
	delete framebufferquad; framebufferquad = nullptr;
	delete depthbufferquad; depthbufferquad = nullptr;
	delete objectBuffer; objectBuffer = nullptr;
	delete gameLogic; gameLogic = nullptr;
	delete horizontalGlowBuffer; horizontalGlowBuffer = nullptr;
	delete verticalGlowBuffer; verticalGlowBuffer = nullptr;
	delete objectBuffer; objectBuffer = nullptr;
	delete text; text = nullptr;
	delete toast; toast = nullptr;
	delete helpView; helpView = nullptr;
	delete frustum; frustum = nullptr;
}

void LoopHelper::initializeProjection() {

	float ratio = (float)width / (float)height;

	proj = glm::perspective(70.0f, ratio, 0.1f, 250.0f);
	frustum->setCamInternals(70.0f, ratio, 0.1f, 250.0f);
	frustum->setCamDef(camera->position(), camera->pointsTo(), camera->up());

	calculateMatrices();
}

void LoopHelper::draw() {

	// 1. Render depth of scene to texture (from ligth's perspective) (Shader: SimpleDepthShader)
	// 2. Render scene as normal to object and glow buffer

	prepareCurrentSceneObjects();

	// 1. Shadow Map
	util::Shader *depthShader = shaderLoader->get("SimpleDepthShader");
	depthbufferquad->bind();
	for (auto sceneObject : currentSceneObjects) {
		sceneObject->drawCustomShader(depthShader);
	}
	depthbufferquad->unbind();


	// 2. Render Scene
	objectBuffer->bind();
	depthbufferquad->activateTexture(); // activate depth map


	// if wire frame (F3) is activated, draw scene only with lines
	if (displayWireFrame) {
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
	}


	// if transparency (F9) is activated
	if (transparency) {
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		// when transparency is enabled, we need to render the whole skybox
		// before writing the first pixel (as the first pixel can already be transparent
		skybox->draw();
	}
	else {
		glDisable(GL_BLEND);
		glBlendFunc(GL_ZERO, GL_ONE);
	}
	

	// set sampling quality (F4)
	skybox->setSamplingQuality(getMinSampling(), getMagSampling());
	
	// bind skybox to different texture unit as other game objects
	skybox->bindToTexture();

	for (auto sceneObject : currentSceneObjects) {
		sceneObject->setSamplingQuality(getMinSampling(), getMagSampling());
		sceneObject->draw();
	}

	
	// prepare toast
	if (gameLogic->isGameOver()) {
		toast->notify("GAME OVER");
	}

	if (gameLogic->isGameWon()){
		toast->notify("YOU WON!");
	}

	// render text
	gameLogic->draw(); // time, mice, life
	toast->draw();

	if (!gameStarted) {
		std::string message = "Press 'S' to start the game...";
		text->scale = 1.5f;
		float leftShift = message.length() * 0.5f * 0.025f; // per character 0.025 of screen length left
		text->draw(message, 0.45f - leftShift, 0.6f);
		text->scale = 1.0f;
	}
	if (gamePause) {
		std::string message = "GAME PAUSED";
		text->scale = 1.5f;
		float leftShift = message.length() * 0.5f * 0.025f; // per character 0.025 of screen length left
		text->draw(message, 0.45f - leftShift, 0.45f);
		text->scale = 1.0f;
	}
	if (displayFrameTime) {
		renderFPS();
	}
	if (displayHelp){
		helpView->draw();
	}


	if (!transparency) {
		// if transparency is not enabled, we can draw the skybox last to avoid drawing it for every pixel
		skybox->draw();
	}

	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	objectBuffer->unbind();

	// currently no objects glow (SceneObject glow = false)
	renderGlow();

	// render frame buffer with object buffer and glow buffer
	framebufferquad->bind();

	// Enabling blending let's us draw the object buffer ON TOP OF the glow buffer.
	// This results in an additive image of both the object buffer and the glow buffer.
	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE, GL_ONE);

	objectBuffer->draw();
	horizontalGlowBuffer->draw();
	verticalGlowBuffer->draw();
	//depthbufferquad->draw(); // only for debugging (renders depth map)

	framebufferquad->unbindAndDraw();
}

void LoopHelper::renderGlow(){
	// render only glowing elements - HORIZONTAL
	horizontalGlowBuffer->bind();
	for (auto sceneObject : currentSceneObjects) {

		if (sceneObject->glowing == true){
			glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
		}
		else {
			glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
		}
		sceneObject->draw();

		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	}

	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	horizontalGlowBuffer->unbind();

	// render only glowing elements - VERTICAL
	verticalGlowBuffer->bind();
	for (auto sceneObject : currentSceneObjects) {
		
		if (sceneObject->glowing){
			glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
		}
		else {
			glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
		}
		sceneObject->draw();

		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	}

	glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	verticalGlowBuffer->unbind();

}




void LoopHelper::update(float time_delta) {
	if (gameStarted 
		&& !gamePause 
		&& !gameLogic->isGameOver()
		&& !gameLogic->isGameWon()) {

		for (auto sceneObject : currentSceneObjects) {
			sceneObject->update(time_delta);
		}
		gameLogic->update(time_delta);
		toast->update(time_delta);
		camera->update();

		fps = (int)(1.0f / time_delta);
		frustum->setCamDef(camera->position(), camera->pointsTo(), camera->up());
		calculateMatrices();
	}

	// set values for restarting game
	if (gameStarted && (gameLogic->isGameOver() || gameLogic->isGameWon()) ) {
		gameStarted = false;
		camera->toggleHeroAttachment();

		// game won
		if (gameLogic->isGameWon()){
			sceneObjectLoader->getHeroParts()->heroBody->rotate(Direction::RIGHT, 180.0f);
		}
	}
	
}

void LoopHelper::updateViewProjections() {

	// update shader
	for (auto pair : shaderLoader->shaders) {
		Shader* shader = pair.second;
		shader->useShader();

		auto proj_location = glGetUniformLocation(shader->programHandle, "proj");
		glUniformMatrix4fv(proj_location, 1, GL_FALSE, glm::value_ptr(proj));

		auto view_proj_location = glGetUniformLocation(shader->programHandle, "view_proj");
		glUniformMatrix4fv(view_proj_location, 1, GL_FALSE, glm::value_ptr(view_proj));

		auto view_location = glGetUniformLocation(shader->programHandle, "view");
		glUniformMatrix4fv(view_location, 1, GL_FALSE, glm::value_ptr(glm::inverse(view)));

		auto cameraPos = camera->position();
		glUniform3f(glGetUniformLocation(shader->programHandle, "cameraPos"),
			cameraPos.x, cameraPos.y, cameraPos.z);
		auto eyeDir = camera->eyeDirection();
		glUniform3f(glGetUniformLocation(shader->programHandle, "eyeDir"),
			eyeDir.x, eyeDir.y, eyeDir.z);


		// - shadow maps
		auto shadowMap_location = glGetUniformLocation(shader->programHandle, "shadowMap");
		glUniform1i(shadowMap_location, 3);

		auto lightSpace_location = glGetUniformLocation(shader->programHandle, "lightSpaceMatrix");
		glUniformMatrix4fv(lightSpace_location, 1, GL_FALSE, glm::value_ptr(lightSpaceMatrix));
		

		// light
		auto lamp = sceneObjectLoader->getLight();

		auto lightPos = lamp->getLightPos();
		auto lightPos_location = glGetUniformLocation(shader->programHandle, "lightPos");
		glUniform3fv(lightPos_location, 1, glm::value_ptr(lightPos));

		auto lightDir = lamp->getLightDir();
		auto lightDir_location = glGetUniformLocation(shader->programHandle, "lightDir");
		glUniform3fv(lightDir_location, 1, glm::value_ptr(lightDir));

		auto lightColor = lamp->getLightColor();
		auto lightColor_location = glGetUniformLocation(shader->programHandle, "lightColor");
		glUniform3fv(lightColor_location, 1, glm::value_ptr(lightColor));

	}

	// update skybox shader to not include camera translation
	shaderLoader->get("SkyBox")->useShader();
	auto view_proj_location = glGetUniformLocation(shaderLoader->get("SkyBox")->programHandle, "view_proj");
	auto view_no_translation = glm::mat4(glm::mat3(camera->modelMatrix));
	auto view_proj_no_translation = proj * view_no_translation;
	glUniformMatrix4fv(view_proj_location, 1, GL_FALSE, glm::value_ptr(view_proj_no_translation));


	// update water shader
	shaderLoader->get("Water")->useShader();
	auto time_location = glGetUniformLocation(shaderLoader->get("Water")->programHandle, "time");
	glUniform1f(time_location, (float)(glfwGetTime()*10.0f));
}

void LoopHelper::calculateMatrices() {
	view = camera->modelMatrix;
	view_proj = proj * view;

	auto lamp = sceneObjectLoader->getLight();
	auto lampPos = lamp->getLightPos();

	float dist = 80.0f;

	//// Box with all objects which should generate shadows: left,right,bottom,top,zNear,zFar
	//auto lightProjection = glm::ortho(-dist, dist, -dist, dist,frustum->getNear(), frustum->getFar()); 
	auto lightProjection = glm::ortho(-dist, dist, -dist, dist, -5.0f, 80.0f);
	auto lightView = glm::lookAt(lampPos, glm::vec3(0.0f, 0.0f, 0.0f), glm::vec3(0, 1, 0));
	lightSpaceMatrix = lightProjection * lightView;

	updateViewProjections();
}


// renders only objects inside Frustum
void LoopHelper::prepareCurrentSceneObjects() {
	currentSceneObjects.clear();

	for (auto sceneObject : sceneObjectLoader->getSceneObjects()) {

		if (viewFrustumCulling) {
			if (frustum->inFrustum(sceneObject) != Frustum::OUTSIDE) {
				currentSceneObjects.push_back(sceneObject);
			}
		}
		else {
			currentSceneObjects.push_back(sceneObject);
		}
	}
}


int LoopHelper::getMagSampling() {
	if (textureSamplingQuality == TextureSamplingQuality::TSQ_BILINEAR) {
		return GL_LINEAR;
	}
	else {
		return GL_NEAREST;
	}
}

int LoopHelper::getMinSampling() {
	if (mipMapQuality == MipMapQuality::MMQ_OFF) {
		if (textureSamplingQuality == TextureSamplingQuality::TSQ_BILINEAR) {
			return GL_LINEAR;
		}
		else if (textureSamplingQuality == TextureSamplingQuality::TSQ_NEAREST_NEIGHBOR) {
			return GL_NEAREST;
		}
	}
	else if (mipMapQuality == MipMapQuality::MMQ_NEAREST_NEIGHBOR) {
		if (textureSamplingQuality == TextureSamplingQuality::TSQ_BILINEAR) {
			return GL_LINEAR_MIPMAP_NEAREST;
		}
		else if (textureSamplingQuality == TextureSamplingQuality::TSQ_NEAREST_NEIGHBOR) {
			return GL_NEAREST_MIPMAP_NEAREST;
		}
	}
	else if (mipMapQuality == MipMapQuality::MMQ_Linear) {
		if (textureSamplingQuality == TextureSamplingQuality::TSQ_BILINEAR) {
			return GL_LINEAR_MIPMAP_LINEAR;
		}
		else if (textureSamplingQuality == TextureSamplingQuality::TSQ_NEAREST_NEIGHBOR) {
			return GL_NEAREST_MIPMAP_LINEAR;
		}
	}

	return GL_LINEAR;
}


void LoopHelper::renderFPS() {
	text->scale = 0.5f;
	std::string fpsT = "FPS: " + std::to_string(fps);
	text->draw(fpsT, 0.9f, 0.001f);
	text->scale = 1.0f;
}

void LoopHelper::startGame() {
	gameStarted = true;
	gameLogic->resetGame();
	camera->attachToHero();
}

bool LoopHelper::isGameStarted() {
	return gameStarted;
}

void LoopHelper::pauseGame(){
	gamePause = true;
}

void LoopHelper::unpauseGame(){
	gamePause = false;
}