#include "InputHandler.hpp"

using namespace cgue::loop;
using namespace cgue::scene;
using namespace cgue::util;
using namespace cgue::loader;

#define MOUSE_SPEED 0.01f
#define MOVEMENT_SPEED 5.0f
#define ROTATION_SPEED 150.0f
#define JUMP_SPEED 5.0f
#define MAX_JUMP_HEIGHT 3.0f

#define HERO_MAX_X 24.0f
#define HERO_MAX_Z 24.0f
#define HERO_MIN_Y 0.0f // 2 = start position, 0 = water


Camera* InputHandler::camera = nullptr;
HeroParts* InputHandler::heroParts = nullptr;
LoopHelper* InputHandler::loopHelper = nullptr;
SoundHandler* InputHandler::soundhandler = nullptr;

bool InputHandler::gameRunning = false;

InputHandler::InputHandler(GLFWwindow* _window, LoopHelper* _loopHelper, SoundHandler* _soundhandler)
:window(_window){
	InputHandler::loopHelper = _loopHelper;
	InputHandler::camera = _loopHelper->camera;
	InputHandler::heroParts = _loopHelper->sceneObjectLoader->getHeroParts();
	soundhandler = _soundhandler;
	registerCallbacks();

	int width, height;
	glfwGetFramebufferSize(window, &width, &height);

	widthHalf = width / 2;
	heightHalf = height / 2;

	jump = false;
	startHeight = heroParts->heroBody->position().y;
	fallingDown = false;
	moveTailUp = true;

	glfwSetInputMode(window, GLFW_CURSOR, GLFW_CURSOR_HIDDEN);
}

InputHandler::~InputHandler() {
}

void InputHandler::processKeyboardInput(float time_delta) {

	if (!gameRunning) {
		return;
	}

	float translationSpeed = MOVEMENT_SPEED * time_delta;
	float rotationSpeed = ROTATION_SPEED * time_delta;
	float jumpspeed = JUMP_SPEED * time_delta;

	glm::vec3 translation = glm::vec3(0.0f);
	float angle = 0.0f;
	Direction dir = LEFT; // default value

	SceneObject* hero = heroParts->heroBody;
	SceneObject* currentPathObj = loopHelper->gameLogic->getCurrentPathObject();

	// ### GET INPUTS ###

	// jump
	if (glfwGetKey(window, GLFW_KEY_SPACE) == GLFW_PRESS && !fallingDown) {

		if (!jump) {
			startHeight = hero->position().y;
			jump = true;
		}
		float jumpHeight = hero->position().y - startHeight;
		
		if (jumpHeight < MAX_JUMP_HEIGHT) {
			translation += glm::vec3(0.0f, jumpspeed, 0.0f);
		}
		// hero reached max height
		else {
			fallingDown = true;
			jump = false;
		}

	}

	// falling down - hero is in pool and not on path
	else if (heroInBoundingBox() && (!loopHelper->gameLogic->isOnPath() || hero->position().y > currentPathObj->height)) {
		fallingDown = true;
	}
	// hero is not in pool
	else if (!heroInBoundingBox() && 
		(!loopHelper->gameLogic->isOnPath() && hero->position().y > 1.5f
		|| loopHelper->gameLogic->isOnPath() && hero->position().y > currentPathObj->height)) {
		fallingDown = true;
	}
	else {
		fallingDown = false;
	}

	// navigate
	// left
	if (glfwGetKey(window, GLFW_KEY_LEFT) == GLFW_PRESS) {
		angle += rotationSpeed;
		dir = LEFT;
	}
	// right
	if (glfwGetKey(window, GLFW_KEY_RIGHT) == GLFW_PRESS) {
		angle -= rotationSpeed;
		dir = RIGHT;
	}
	// forward
	if (glfwGetKey(window, GLFW_KEY_UP) == GLFW_PRESS) {
		translation += glm::vec3(0.0f, 0.0f, translationSpeed);
	}


	// ### PROCESS INPUTS ###

	// falling down
	if (fallingDown) {
		translation -= glm::vec3(0.0f, jumpspeed, 0.0f);
	}

	// rotation
	if (angle != 0.0f) {
		rotateHeroParts(dir, angle);
	}

	// translate
	if (translation != glm::vec3(0.0f)) {
		translateHeroParts(translation);

		// revert translation
		//if (!heroInBoundingBox()) { // hero is no longer in bounding box
		//	translateHeroParts(-translation);
		//}
		//else 
		if (hero->detectHit(currentPathObj) 
			&& hero->detectFrontHit(currentPathObj)) { // hero runs into path object
			translateHeroParts(glm::vec3(0.0f, 0.0f, -translation.z)); // revert translation
			
			if (hero->position().y+hero->height > currentPathObj->position().y) { // near top of path object - push up
				translateHeroParts(glm::vec3(0.0f, 0.1f, 0.0f));
			}
		}
		
	}

	moveTail(translationSpeed*0.5);
	getRotationFromMouse(time_delta); // process mouse input
}

void InputHandler::translateHeroParts(glm::vec3 trans) {
	heroParts->heroBody->translate(trans);

	heroParts->heroTail->modelMatrix = heroParts->heroBody->modelMatrix;
	heroParts->heroTail->translate(heroParts->tailBodyDifference);
	heroParts->heroTail->translate(trans);
}

void InputHandler::rotateHeroParts(Direction dir, float angle) {
	glm::mat4 old_body_m = heroParts->heroBody->modelMatrix;

	heroParts->heroBody->rotate(dir, angle);

	heroParts->heroTail->modelMatrix = old_body_m;
	heroParts->heroTail->rotate(dir, angle);
	heroParts->heroTail->translate(heroParts->tailBodyDifference);
	heroParts->heroTail->rotate(dir, heroParts->tailAngle);

}

void InputHandler::moveTail(float translationSpeed) {
	if (moveTailUp) {
		heroParts->tailAngle += 10.0f * translationSpeed; 
		if (heroParts->tailAngle > 8.0f) {
			moveTailUp = false;
		}
	}
	else {
		heroParts->tailAngle -= 10.0f * translationSpeed;
		if (heroParts->tailAngle < -8.0f) {
			moveTailUp = true;
		}
	}
}



bool InputHandler::isGameRunning() {
	return gameRunning;
}

void InputHandler::getRotationFromMouse(float time_delta) {
	glfwGetCursorPos(window, &mouseX, &mouseY);
	glfwSetCursorPos(window, widthHalf, heightHalf);

	float horizontalAngle = MOUSE_SPEED * float(widthHalf - mouseX);
	float verticalAngle = MOUSE_SPEED * float(heightHalf - mouseY);

	
	rotateHeroParts(Direction::LEFT, horizontalAngle);
	camera->rotateCameraHorizontal(verticalAngle);
}

void InputHandler::mouseScrollCallback(GLFWwindow* window, double xoffset, double yoffset) {
	if (yoffset > 0) {
		camera->zoomIn();
	}
	else {
		camera->zoomOut();
	}
}


void InputHandler::registerCallbacks() {
	glfwSetScrollCallback(window, mouseScrollCallback);
	glfwSetKeyCallback(window, keyCallback);
}

void InputHandler::keyCallback(GLFWwindow* window, int key, int scancode, int action, int mode) {
	if (action == GLFW_PRESS) {

		// start game
		if (key == GLFW_KEY_S) {
			if (!loopHelper->isGameStarted()) {
				gameRunning = true;
				loopHelper->startGame();
				soundhandler->playSound();
			}
		}

		// pause game
		if (key == GLFW_KEY_P) {
			if (gameRunning) {
				gameRunning = false;
				loopHelper->pauseGame();
				soundhandler->mute();
			}
			else {
				gameRunning = true;
				loopHelper->unpauseGame();
				soundhandler->mute();
			}
		}

		// only save keybord input if game is running
		if (gameRunning) {
			glfwSetInputMode(window, GLFW_STICKY_KEYS, GL_TRUE);
		}
		else {
			glfwSetInputMode(window, GLFW_STICKY_KEYS, GL_FALSE); 
		}


		// toggle camera attachment to hero
		if (key == GLFW_KEY_E) {
			camera->toggleHeroAttachment();
		}

		// mute sound
		if (key == GLFW_KEY_M) {
			soundhandler->mute();
		}

		// play moving sound
		if ((key == GLFW_KEY_LEFT) || (key == GLFW_KEY_RIGHT)) {
			if (gameRunning) {
				soundhandler->playMove();
			}
		}

		// help view (hide/show)
		if (key == GLFW_KEY_F1) {
			if (loopHelper->displayHelp) {
				loopHelper->displayHelp = false;
			}
			else {
				loopHelper->displayHelp = true;
			}
		}

		// frames per second (hide/show)
		if (key == GLFW_KEY_F2) {
			if (loopHelper->displayFrameTime) {
				loopHelper->displayFrameTime = false;
				loopHelper->toast->notify("Frame Time Disabled");
			}
			else {
				loopHelper->displayFrameTime = true;
				loopHelper->toast->notify("Frame Time Enabled");
			}
		}

		// render scene only with lines (en-/disable)
		if (key == GLFW_KEY_F3) {
			if (loopHelper->displayWireFrame) {
				loopHelper->displayWireFrame = false;
				loopHelper->toast->notify("Wire Frame Disabled");
			}
			else {
				loopHelper->displayWireFrame = true;
				loopHelper->toast->notify("Wire Frame Enabled");
			}
		}

		// texture sampling quality: bilinear/nearest neighbor
		if (key == GLFW_KEY_F4) {
			if (loopHelper->textureSamplingQuality == TextureSamplingQuality::TSQ_BILINEAR) {
				loopHelper->textureSamplingQuality = TextureSamplingQuality::TSQ_NEAREST_NEIGHBOR;
				loopHelper->toast->notify("Texture Sampling to NEAREST");
			}
			else if (loopHelper->textureSamplingQuality == TextureSamplingQuality::TSQ_NEAREST_NEIGHBOR) {
				loopHelper->textureSamplingQuality = TextureSamplingQuality::TSQ_BILINEAR;
				loopHelper->toast->notify("Texture Sampling to BILINEAR");
			}
		}

		// toggle mip map qualities
		if (key == GLFW_KEY_F5){
			if (loopHelper->mipMapQuality == MipMapQuality::MMQ_Linear) {
				loopHelper->mipMapQuality = MipMapQuality::MMQ_NEAREST_NEIGHBOR;
				loopHelper->toast->notify("Mip Map Quality NEAREST");
			}
			else if (loopHelper->mipMapQuality == MipMapQuality::MMQ_NEAREST_NEIGHBOR) {
				loopHelper->mipMapQuality = MipMapQuality::MMQ_OFF;
				loopHelper->toast->notify("Mip Map Quality OFF");
			}
			else if (loopHelper->mipMapQuality == MipMapQuality::MMQ_OFF) {
				loopHelper->mipMapQuality = MipMapQuality::MMQ_Linear;
				loopHelper->toast->notify("Mip Map Quality LINEAR");
			}

		}

		// normal mapping (en-/disable)
		if (key == GLFW_KEY_F6) {
			const std::vector<SceneObject*> sceneObjects = loopHelper->sceneObjectLoader->getSceneObjects();

			for (SceneObject* sceneObject : sceneObjects) {
				sceneObject->changeNormalMapping();
			}
			loopHelper->toast->notify("Normal Mapping ON/OFF");

		}

		// view frustum culling (en-/disable)
		if (key == GLFW_KEY_F8) {
			if (loopHelper->viewFrustumCulling) {
				loopHelper->viewFrustumCulling = false;
				loopHelper->toast->notify("View Frustum Culling Disabled");
			}
			else {
				loopHelper->viewFrustumCulling = true;
				loopHelper->toast->notify("View Frustum Culling Enabled");
			}
		}

		// transparency (en-/disable)
		if (key == GLFW_KEY_F9) {
			if (loopHelper->transparency) {
				loopHelper->transparency = false;
				loopHelper->toast->notify("Transparency disabled");
			}
			else {
				loopHelper->transparency = true;
				loopHelper->toast->notify("Transparency enabled");
			}
		}


	}
}

bool InputHandler::heroInBoundingBox() {
	auto pos = heroParts->heroBody->position();

	if (std::abs(pos.x) > HERO_MAX_X) {
		return false;
	}
	else if (std::abs(pos.z) > HERO_MAX_Z) {
		return false;
	}
	return true;
}