#include "GameLogic.hpp"
#include <time.h>

using namespace cgue::loop;
using namespace cgue::loader;
using namespace cgue::scene;

#define GAME_STARTING_POINTS 0
#define GAME_STARTING_TIME 90
#define GAME_LIFE_STARTING 3

GameLogic::GameLogic(SceneObjectLoader* _sceneObjectLoader, Text* _text, SoundHandler* _soundhandler)
: sceneObjectLoader(_sceneObjectLoader), text(_text), soundHandler(_soundhandler) {
	
	miceLeft = sceneObjectLoader->getNumGood();
	lifeLeft = GAME_LIFE_STARTING;
	timeInSeconds = GAME_STARTING_TIME;

	onPath = true;
	inWater = false;
	heroStartPos = sceneObjectLoader->getHeroParts()->heroBody->position();
	currentPathObj = sceneObjectLoader->getStartPathObject();
}

void GameLogic::resetGame() {

	sceneObjectLoader->resetObjects();

	miceLeft = sceneObjectLoader->getNumGood();
	lifeLeft = GAME_LIFE_STARTING;
	timeInSeconds = GAME_STARTING_TIME;

	onPath = true;
	inWater = false;
	heroStartPos = sceneObjectLoader->getHeroParts()->heroBody->position();
	currentPathObj = sceneObjectLoader->getStartPathObject();

}


void GameLogic::update(double time_delta) {

	timeInSeconds -= time_delta;

	SceneObject* hero = sceneObjectLoader->getHeroParts()->heroBody;

	onPath = false;

	for (SceneObject* sceneObject : sceneObjectLoader->getSceneObjects()) {
		if (sceneObject->objectType == ObjectType::GOOD ||
			sceneObject->objectType == ObjectType::BAD ||
			sceneObject->objectType == ObjectType::BONUS1 ||
			sceneObject->objectType == ObjectType::BONUS2) {

			// hero hits a mouse or bonus
			if (hero->detectHit(sceneObject)) {
				processHit(sceneObject);
				sceneObjectLoader->remove(sceneObject);
			}
		}
		
		if (sceneObject->objectType == ObjectType::PATH) {
			
			// hero hits pool object
			if (hero->detectHit(sceneObject)) {
				currentPathObj = sceneObject;
				onPath = true;
			}
		}

	}

	if (onPath == false && hero->position().y <= 0.0f) {
		inWater = true;
		lifeLeft = 0;
		soundHandler->playSplash();
	}

	// play sound
	if (isGameOver()) {
		soundHandler->playGameOver();

	}
	else if (isGameWon()) {
		soundHandler->playGameWon();
	}

}

void GameLogic::processHit(SceneObject* sceneObject) {
	if (sceneObject->objectType == ObjectType::GOOD) {
		miceLeft -= 1;
		soundHandler->playGood();
	}
	else if (sceneObject->objectType == ObjectType::BAD) {
		lifeLeft -= 1;
		soundHandler->playBad();
	}
	else if (sceneObject->objectType == ObjectType::BONUS1) {
		lifeLeft += 1;
		soundHandler->playBonus();
	}
	else if (sceneObject->objectType == ObjectType::BONUS2) {
		timeInSeconds += 25.0;
		soundHandler->playBonus();
	}
}

void GameLogic::draw() {

	struct tm newtime;
	char buf[26];

	// convert left seconds into string
	time_t seconds((int)timeInSeconds);
	gmtime_s(&newtime, &seconds);
	asctime_s(buf, 26, &newtime);

	sprintf_s(buf, "Time: %02d:%02d", newtime.tm_min, newtime.tm_sec);
	text->draw(buf, 0.001f, 0.001f);

	std::string miceT = "Mice left: " + std::to_string(miceLeft);
	text->draw(miceT, 0.001f, 0.05f);

	std::string lifeT = "Life left: " + std::to_string(lifeLeft);
	text->draw(lifeT, 0.001f, 0.10f);
}

bool GameLogic::isOnPath() {
	return onPath;
}

bool GameLogic::isGameOver() {
	return ((timeInSeconds <= 0) || (lifeLeft <= 0) || (inWater == true));
}

bool GameLogic::isGameWon() {
	return miceLeft <= 0;
}

SceneObject* GameLogic::getCurrentPathObject() {
	return currentPathObj;
}