#include "Frustum.hpp"
#include <iostream>
#include <string>

using namespace cgue::loop;
using namespace cgue::scene;

Frustum::Frustum() {
	nearD = 0.0f;
	farD = 0.0f;
	ratio = 0.0f;
	angle = 0.0f;
	tang = 0.0f;

	nw = 0.0f;
	nh = 0.0f;
	fw = 0.0f;
	fh = 0.0f;

	/*ntl = glm::vec3(0.0f, 0.0f, 0.0f);
	ntr = glm::vec3(0.0f, 0.0f, 0.0f);
	nbl = glm::vec3(0.0f, 0.0f, 0.0f);
	nbr = glm::vec3(0.0f, 0.0f, 0.0f);
	ftl = glm::vec3(0.0f, 0.0f, 0.0f);
	ftr = glm::vec3(0.0f, 0.0f, 0.0f);
	fbl = glm::vec3(0.0f, 0.0f, 0.0f);
	fbr = glm::vec3(0.0f, 0.0f, 0.0f);*/

	// the six view frustum planes
	pl[TOP] = new Plane();
	pl[BOTTOM] = new Plane();
	pl[LEFT] = new Plane();
	pl[RIGHT] = new Plane();
	pl[NEARP] = new Plane();
	pl[FARP] = new Plane();

}

Frustum::~Frustum() {
	for (auto p : pl){
		delete p; p = nullptr;
	}
}

/*
	angle... viewing angle
	ratio... aspect ratio
	nearD... distance to near plane
	farD... distance to far plane
*/

void Frustum::setCamInternals(float _angle, float _ratio, float _nearD, float _farD) {
	// store the information
	ratio = _ratio;
	angle = _angle;
	nearD = _nearD;
	farD = _farD;

	// compute width and height of the near and far plane sections
	tang = (float)tan(glm::radians(angle * 0.5f));
	nh = nearD * tang;
	nw = nh * ratio;
	fh = farD  * tang;
	fw = fh * ratio;
}


/*
	p...	camera position
	l...	camera eye direction
	up...	up vector of camera
	right...right vector of camera
*/
void Frustum::setCamDef(glm::vec3 &p, glm::vec3 &l, glm::vec3 &up) {
	
	glm::vec3 nc, fc, X, Y, Z;

	nc = glm::vec3(0.0f);
	fc = glm::vec3(0.0f);
	Z = glm::vec3(0.0f);

	// compute the Z axis of camera
	// this axis points in the opposite direction from the looking direction
	Z = p - l;
	Z = glm::normalize(Z);
	
	// X axis of camera with given "up" vector and Z axis
	X = glm::cross(up, Z);
	X = glm::normalize(X);

	// the real "up" vector is the cross product of Z and X
	Y = glm::cross(Z, X);


	// compute the centers of the near and far planes
	nc = p - Z * nearD;
	fc = p - Z * farD;

	//// compute the 4 corners of the frustum on the near plane
	//ntl = nc + Y * nh - X * nw;
	//ntr = nc + Y * nh + X * nw;
	//nbl = nc - Y * nh - X * nw;
	//nbr = nc - Y * nh + X * nw;

	//// compute the 4 corners of the frustum on the far plane
	//ftl = fc + Y * fh - X * fw;
	//ftr = fc + Y * fh + X * fw;
	//fbl = fc - Y * fh - X * fw;
	//fbr = fc - Y * fh + X * fw;

	// NEAR, FAR
	pl[NEARP]->setNormalAndPoint(-Z, nc);
	pl[FARP]->setNormalAndPoint(Z, fc);

	glm::vec3 aux, normal;

	// TOP
	aux = (nc + Y*nh) - p;
	aux = glm::normalize(aux);
	normal = glm::cross(aux, X);
	pl[TOP]->setNormalAndPoint(normal, nc + Y*nh);

	// BOTTOM
	aux = (nc - Y*nh) - p;
	aux = glm::normalize(aux);
	normal = glm::cross(X, aux);
	pl[BOTTOM]->setNormalAndPoint(normal, nc - Y*nh);

	// LEFT
	aux = (nc - X*nw) - p;
	aux = glm::normalize(aux);
	normal = glm::cross(aux, Y);
	pl[LEFT]->setNormalAndPoint(normal, nc - X*nw);

	// RIGHT
	aux = (nc + X*nw) - p;
	aux = glm::normalize(aux);
	normal = glm::cross(Y, aux);
	pl[RIGHT]->setNormalAndPoint(normal, nc + X*nw);

}

int Frustum::inFrustum(SceneObject* sceneObject) {
	float distance;
	int result = INSIDE;

	for (int i = 0; i < 6; i++) {
		distance = pl[i]->distance(sceneObject->position());
		if (distance < -sceneObject->radius) {
			return OUTSIDE;
		}
		else if (distance < sceneObject->radius) {
			result = INTERSECT;
		}
	}
	return result;
}


float Frustum::getWidth() {
	return fw;
}

float Frustum::getHeight() {
	return fh;
}

//near distance
float Frustum::getNear() {
	return nearD;
}

//far distance
float Frustum::getFar() {
	return farD;
}