#include "DebugContext.hpp"

using namespace cgue::loop;

DebugContext::DebugContext() {
#if _DEBUG
	// Create a debug OpenGL context or tell your OpenGL library (GLFW, SDL) to do so.
	glfwWindowHint(GLFW_OPENGL_DEBUG_CONTEXT, GL_TRUE);

	//this->registerDebugCallback();
#endif
}

void DebugContext::registerDebugCallback() {
#if _DEBUG
	// Query the OpenGL function to register your callback function.
	PFNGLDEBUGMESSAGECALLBACKPROC _glDebugMessageCallback = (PFNGLDEBUGMESSAGECALLBACKPROC)glfwGetProcAddress("glDebugMessageCallback");
	PFNGLDEBUGMESSAGECALLBACKARBPROC _glDebugMessageCallbackARB = (PFNGLDEBUGMESSAGECALLBACKARBPROC)glfwGetProcAddress("glDebugMessageCallbackARB");
	PFNGLDEBUGMESSAGECALLBACKAMDPROC _glDebugMessageCallbackAMD = (PFNGLDEBUGMESSAGECALLBACKAMDPROC)glfwGetProcAddress("glDebugMessageCallbackAMD");

	// Register your callback function.
	if (_glDebugMessageCallback != NULL) {
		_glDebugMessageCallback(DebugContext::DebugCallback, NULL);
	}
	if (_glDebugMessageCallbackARB != NULL) {
		_glDebugMessageCallbackARB(DebugContext::DebugCallback, NULL);
	}
	if (_glDebugMessageCallbackAMD != NULL) {
		_glDebugMessageCallbackAMD(DebugContext::DebugCallbackAMD, NULL);
	}

	// Enable synchronous callback. This ensures that your callback function is called
	// right after an error has occurred. This capability is not defined in the AMD
	// version.
	if ((_glDebugMessageCallback != NULL) || (_glDebugMessageCallbackARB != NULL)) {
		glEnable(GL_DEBUG_OUTPUT_SYNCHRONOUS);
	}
#endif
}

std::string DebugContext::FormatDebugOutput(GLenum source, GLenum type, GLuint id, GLenum severity, const char* msg) {
	std::stringstream stringStream;
	std::string sourceString;
	std::string typeString;
	std::string severityString;

	// The AMD variant of this extension provides a less detailed classification of the error,
	// which is why some arguments might be "Unknown".
	switch (source) {
	
		case GL_DEBUG_CATEGORY_API_ERROR_AMD:
		case GL_DEBUG_SOURCE_API: {
									sourceString = "API";
									break;
		}
		case GL_DEBUG_CATEGORY_APPLICATION_AMD:
		case GL_DEBUG_SOURCE_APPLICATION: {
									sourceString = "Application";
									break;
		}
		case GL_DEBUG_CATEGORY_WINDOW_SYSTEM_AMD:
		case GL_DEBUG_SOURCE_WINDOW_SYSTEM: {
									sourceString = "Window System";
									break;
		}
		case GL_DEBUG_CATEGORY_SHADER_COMPILER_AMD:
		case GL_DEBUG_SOURCE_SHADER_COMPILER: {
									sourceString = "Shader Compiler";
									break;
		}
		case GL_DEBUG_SOURCE_THIRD_PARTY: {
									sourceString = "Third Party";
									break;
		}
		case GL_DEBUG_CATEGORY_OTHER_AMD:
		case GL_DEBUG_SOURCE_OTHER: {
									sourceString = "Other";
									break;
		}
		default: {
					 sourceString = "Unknown";
					 break;
		}
	}

	switch (type) {
		case GL_DEBUG_TYPE_ERROR: {
									typeString = "Error";
									break;
		}
		case GL_DEBUG_CATEGORY_DEPRECATION_AMD:
		case GL_DEBUG_TYPE_DEPRECATED_BEHAVIOR: {
									typeString = "Deprecated Behavior";
									break;
		}
		case GL_DEBUG_CATEGORY_UNDEFINED_BEHAVIOR_AMD:
		case GL_DEBUG_TYPE_UNDEFINED_BEHAVIOR: {
									typeString = "Undefined Behavior";
									break;
		}
		case GL_DEBUG_TYPE_PORTABILITY_ARB: {
									typeString = "Portability";
									break;
		}
		case GL_DEBUG_CATEGORY_PERFORMANCE_AMD:
		case GL_DEBUG_TYPE_PERFORMANCE: {
									typeString = "Performance";
									break;
		}
		case GL_DEBUG_CATEGORY_OTHER_AMD:
		case GL_DEBUG_TYPE_OTHER: {
									typeString = "Other";
									break;
		}
		default: {
						typeString = "Unknown";
						break;
		}
	}

	switch (severity) {
		case GL_DEBUG_SEVERITY_HIGH: {
									severityString = "High";
									break;
		}
		case GL_DEBUG_SEVERITY_MEDIUM: {
									severityString = "Medium";
									break;
		}
		case GL_DEBUG_SEVERITY_LOW: {
									severityString = "Low";
									break;
		}
		default: {
					 severityString = "Unknown";
					 break;
		}
	}

	stringStream << "OpenGL Error: " << msg;
	stringStream << " [Source = " << sourceString;
	stringStream << ", Type = " << typeString;
	stringStream << ", Severity = " << severityString;
	stringStream << ", ID = " << id << "]";

	return stringStream.str();
}

void APIENTRY DebugContext::DebugCallback(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const GLvoid* userParam) {
	switch (id) {
		case 131218: {
						 // Program/shader state performance warning:
						 // Fragment Shader is going to be recompiled because the shader key based on GL state mismatches.
						 return;
		}
		case 131185: {
						 // Buffer detailed info: Buffer object 1 (bound to GL_ARRAY_BUFFER_ARB, usage hint is GL_STATIC_DRAW)
						 // will use VIDEO memory as the source for buffer object operations.
						 return;
		}
		default: {
					 break;
		}
	}
	std::string error = DebugContext::FormatDebugOutput(source, type, id, severity, message);
	std::cout << error << std::endl;
}

void APIENTRY DebugContext::DebugCallbackAMD(GLuint id, GLenum category, GLenum severity, GLsizei length, const GLchar* message, GLvoid* userParam) {
	std::string error = DebugContext::FormatDebugOutput(category, category, id, severity, message);
	std::cout << error << std::endl;
}