#include "Spawner.hpp"

using namespace cgue::loader;
using namespace cgue::util;
using namespace cgue::scene;
using namespace std;

#define SPAWNER_MAX_X  21.0f
#define SPAWNER_MIN_X -21.0f
#define SPAWNER_MIN_Y  5.0f
#define SPAWNER_MAX_Y -5.0f
#define SPAWNER_MIN_Z  21.0f
#define SPAWNER_MAX_Z -21.0f
#define SPAWNER_HERO_RADIUS 1.0f
#define SPAWNER_GOOD_COUNT 6
#define SPAWNER_BAD_COUNT 6
#define SPAWNER_PATH_COUNT 11
#define SPAWNER_BONUS_COUNT 3


Spawner::Spawner(ShaderLoader* _shaderLoader) : shaderLoader(_shaderLoader) {
	// seed the random generator
	std::srand(std::time(0));
	vectorPosition = 0;
	duckHeadPos = glm::vec3(0.0f);

	spawnObjects();
}

/**
* Spawns all objects that are crucial for the game
*/
void Spawner::spawnObjects() {

	int i;

	spawnBigBallAt(glm::vec3(0.0f, 0.0f, 0.0f)); // start (ball in shadow when rendering after all)
	spawnDuckAt(randomInPool()); // 1 Duck per Game
	spawnFurniture();

	for (i = 1; i < SPAWNER_PATH_COUNT; i++) {
		spawnObject(ObjectType::PATH);
	}

	vectorPosition++;
	// mouse on duck head
	spawnMouseAt(duckHeadPos);

	for (i = 0; i < SPAWNER_BONUS_COUNT; i++) {
		spawnObject(ObjectType::BONUS1);
	}

	for (i = 0; i < SPAWNER_BAD_COUNT; i++) {
		spawnObject(ObjectType::BAD);
	}

	for (i = 1; i < SPAWNER_GOOD_COUNT; i++) { // one on duck head
		spawnObject(ObjectType::GOOD);
	}
	

}


void Spawner::spawnFurniture() {
	
	// Deckchairs
	// First
	Model* chair;
	chair = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("Shadow"), "Models//environment//deckchair//deckchair.obj");
	chair->translate(glm::vec3(5.0f, 1.0f, 38.0f));
	chair->rotate(Direction::LEFT, 190.0f);
	chair->scale(glm::vec3(4.0f));
	chair->height = 3.0f;
	chair->radius = 2.5f;
	objects.push_back(chair);
	pathObjects.push_back(chair);

	// Second
	chair = new Model(chair->modelMatrix, ObjectType::PATH, shaderLoader->get("Shadow"), "Models//environment//deckchair//deckchair.obj");
	chair->translate(glm::vec3(-2.0f, 0.0f, 0.0f));
	chair->rotate(Direction::LEFT, 20.0f);
	chair->height = 3.0f;
	chair->radius = 2.5f;
	objects.push_back(chair);
	pathObjects.push_back(chair);

	// Third
	chair = new Model(chair->modelMatrix, ObjectType::PATH, shaderLoader->get("Shadow"), "Models//environment//deckchair//deckchair.obj");
	chair->translate(glm::vec3(-4.0f, 0.0f, 19.0f));
	chair->rotate(Direction::LEFT, 180.0f);
	chair->height = 3.0f;
	chair->radius = 2.5f;
	objects.push_back(chair);
	pathObjects.push_back(chair);


	// Table with Umbrella
	Model *table = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("DiffuseMaterial"), "Models//environment//table//table.obj");
	table->translate(glm::vec3(-6.0f, 6.0f, -32.0f));
	table->scale(glm::vec3(4.0f));
	table->height = 6.3f;
	table->radius = 2.0f;
	objects.push_back(table);

	Model *umrella = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("DiffuseMaterial"), "Models//environment//table//umbrella.obj");
	umrella->translate(glm::vec3(-6.0f, 1.0f, -32.0f));
	umrella->scale(glm::vec3(4.0f));
	umrella->height = 20.0f;
	umrella->radius = 0.25f;
	objects.push_back(umrella);
}


/**
* Spawns a specific type of object. Either PATH, GOOD, BAD or BONUS.
*/
void Spawner::spawnObject(ObjectType objectType) {
	if (objectType == ObjectType::PATH) {
		spawnBigBallAt(randomInPool());
		spawnLifePreserverAt(randomInPool());
		spawnSmallBallAt(randomInPool());
	}
	else if (objectType == ObjectType::GOOD) {
		spawnMouseAt(onPoolPathObject());
	}
	else if (objectType == ObjectType::BAD) {
		spawnPoisonMouseAt(onPoolPathObject());
	}
	else {
		spawnHeartAt(onPoolPathObject());
		spawnClockAt(onPoolPathObject());
	}
}

glm::vec3 Spawner::randomInPool() {
	glm::vec3 pos;

	do {
		pos = glm::vec3(
			random(SPAWNER_MIN_X, SPAWNER_MAX_X),
			0.0f,
			random(SPAWNER_MIN_Z, SPAWNER_MAX_Z)
			);
	} while (collision(pos));

	poolPathObjectPositions.push_back(pos);
	return pos;
}

glm::vec3 Spawner::onPoolPathObject() {
	//glm::vec3 pos = poolPathObjectPositions.at(vectorPosition);
	//pos.y = 2.5f;
	SceneObject* pathObject = pathObjects.at(vectorPosition);
	glm::vec3 pos = pathObject->position();
	pos.y += pathObject->height;

	vectorPosition++;
	return pos;
}

std::vector<Model*> Spawner::getObjects() {
	return objects;
}

const int Spawner::getNumGood() {
	return SPAWNER_GOOD_COUNT;
}


/**
* Returns true when the vector is inside a cube in the center with a radius of SPAWNER_HERO_RADIUS
*/
bool Spawner::inHeroCube(glm::vec3 pos) {
	if (abs(pos.x) < SPAWNER_HERO_RADIUS) {
		return true;
	}
	if (abs(pos.y) < SPAWNER_HERO_RADIUS) {
		return true;
	}
	if (abs(pos.z) < SPAWNER_HERO_RADIUS) {
		return true;
	}
	return false;
}

// objects shouldnt be placed at positions that are already occupied
bool Spawner::collision(glm::vec3 pos){

	for (auto sceneObject : objects) {
		if (sceneObject->objectType == ObjectType::PATH) {

			glm::vec3 posObject = sceneObject->position();
			auto vec = posObject - pos;
			float distance = sqrt(vec.x*vec.x + vec.y*vec.y + vec.z*vec.z);

			if (distance < (3.5f + sceneObject->radius)) { //keep a little distance to other objects
				return true;
			}
		}
	}
	return false;
}


/**
* Generates random floats between min and max
*/
float Spawner::random(float min, float max) {
	return min + static_cast <float> (rand()) / (static_cast <float> (RAND_MAX / (max - min)));
}

void Spawner::spawnMouseAt(glm::vec3 position){
	Model *model = new Model(glm::mat4(1.0f), ObjectType::GOOD, shaderLoader->get("NormalMap"), "Models//mouse//mouse.obj");
	model->translate(position+glm::vec3(0.0f, 0.2f, 0.0f));
	model->scale(glm::vec3(0.5f));
	model->radius = 0.5f;
	model->height = 0.5f;
	model->glowing = true;
	objects.push_back(model);
}

void Spawner::spawnPoisonMouseAt(glm::vec3 position){
	Model* model = new Model(glm::mat4(1.0f), ObjectType::BAD, shaderLoader->get("NormalMapGreen"), "Models//mouse//mouse.obj");
	model->translate(position + glm::vec3(0.0f, 0.2f, 0.0f));
	model->scale(glm::vec3(0.5f));
	model->radius = 0.5f;
	model->height = 0.5f;
	model->glowing = true;
	objects.push_back(model);
}

void Spawner::spawnBigBallAt(glm::vec3 position){
	Model *model = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("Shadow"), "Models//path//ball//rainbowBall.obj");
	model->translate(position);
	model->rotate(Direction::LEFT, 40.0f);
	model->rotate(Direction::DOWN, -35.0f);
	model->scale(glm::vec3(2.5f));
	model->radius = 1.5f;
	model->height = 2.5f;
	
	objects.push_back(model);
	pathObjects.push_back(model);
}

void Spawner::spawnSmallBallAt(glm::vec3 position){
	Model *model = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("Shadow"), "Models//path//ball//polkaDotBall.obj");
	model->translate(position);
	model->scale(glm::vec3(1.5f));
	model->radius = 0.5f;
	model->height = 1.5f;

	objects.push_back(model);
	pathObjects.push_back(model);
}

void Spawner::spawnLifePreserverAt(glm::vec3 position){
	Model* model = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("Shadow"), "Models//path//lifePreserver//lifePreserver.obj");
	model->translate(position);
	model->scale(glm::vec3(2.0f));
	model->radius = 2.0f;
	model->height = 0.5f;

	objects.push_back(model);
	pathObjects.push_back(model);
}

void Spawner::spawnDuckAt(glm::vec3 position){

	// BODY
	Model *model = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("Shadow"), "Models//path//duck//duckBody.obj");
	model->translate(position);
	model->scale(glm::vec3(5.0f));
	model->radius = 3.0f;
	model->height = 2.5f;
	
	objects.push_back(model);
	//pathObjects.push_back(model);

	// save finish position to set mouse
	duckHeadPos = position + glm::vec3(1.0f, 4.5f, -0.1f);

	// HEAD (away from origin: 0.51, 0.10, 1.26)
	model = new Model(glm::mat4(1.0f), ObjectType::PATH, shaderLoader->get("Shadow"), "Models//path//duck//duckHead.obj");
	model->translate(duckHeadPos);
	model->scale(glm::vec3(5.0f));
	model->radius = 1.5f;
	model->height = 7.0f;
	objects.push_back(model);

	duckHeadPos += glm::vec3(1.0f, model->height - model->position().y + 0.1, 0.0f); // for mouse
}

void Spawner::spawnHeartAt(glm::vec3 position) {
	Model* heart = new Model(glm::mat4(1.0f), ObjectType::BONUS1, shaderLoader->get("EnvironmentalMap"), "Models//bonus//heart//heart.obj");
	heart->translate(position);
	heart->translate(glm::vec3(0.0f, 0.5f, 0.0f));
	heart->scale(glm::vec3(1.5f));
	heart->radius =	1.0f;
	heart->height = 1.0f;
	objects.push_back(heart);
}

void Spawner::spawnClockAt(glm::vec3 position) {
	Model* model = new Model(glm::mat4(1.0f), ObjectType::BONUS2, shaderLoader->get("Shadow"), "Models//bonus//clock//clock.obj");
	model->translate(position);
	//model->scale(glm::vec3(0.05f));
	model->radius = 1.0f;
	model->height = 1.0f;

	objects.push_back(model);
}


SceneObject* Spawner::getStartObject() {
	return pathObjects.at(0);
}