//reference: http://learnopengl.com/#!Advanced-OpenGL/Shadows/Shadow-Mapping
#version 330 core

struct Material {
	sampler2D texture_diffuse1;
	sampler2D texture_specular1;
	sampler2D texture_normal1;
	float shininess;
};

struct Light {
    vec3 position;
    vec3 ambient;
    vec3 diffuse;
    vec3 specular;
};

in vec3 Normal;
in vec2 TexCoords;
in vec3 FragPos;
in mat3 TBN;
in vec4 FragPosLightSpace;

out vec4 fragColor;

uniform bool normalMapping;
uniform sampler2D shadowMap;
uniform vec3 cameraPos;
uniform Light light;
uniform Material material;




float ShadowCalculation(vec4 fragPosLightSpace)
{
	
	// perform perspective divide
    vec3 projCoords = fragPosLightSpace.xyz / fragPosLightSpace.w;
    projCoords = projCoords * 0.5f + 0.5f;//transform from [-1,1] to [0,1]
	
	// Keep the shadow at 0.0 when outside the far_plane region of the light's frustum.
	if( projCoords.z > 1.0) {
		return 0.0f;
	}
	
	float closestDepth = texture(shadowMap, projCoords.xy).r; 
	float currentDepth = projCoords.z;
	float shadow = 0.0f;
	
	vec3 normal = normalize(Normal);
	vec3 lightDir = normalize(light.position - FragPos);
	//float bias = max(0.005 * (1.0 - dot(normal, lightDir)), 0.0005);
	float bias = 0.001*tan(acos(dot(normal,lightDir))); 
	bias = clamp(bias, 0,0.01); // clamped between 0 and 1
	
	//pcf calculation
	vec2 texelSize = 1.0 / textureSize(shadowMap, 0);
    
	// sample 9 values around the projected coordinate's x and y value
	for(int x = -1; x <= 1; ++x)
    {
        for(int y = -1; y <= 1; ++y)
        {
            float pcfDepth = texture(shadowMap, projCoords.xy + vec2(x, y) * texelSize).r;
			if (currentDepth - bias > pcfDepth){
				shadow += 1.0;
			}
            //shadow += currentDepth - bias > pcfDepth  ? 1.0 : 0.0;        
        }    
    }
    shadow /= 9.0;
	
	return shadow;	
}




void main()
{	
	vec3 viewDir = normalize(cameraPos - FragPos);
    vec3 lightDir = normalize(light.position - FragPos);
	vec3 normal = normalize(Normal);
	
	if (normalMapping) {
		// obtain normal from normal map in range [0,1], 
		normal = texture(material.texture_normal1, TexCoords).gbr; // rgb, gbr, grb, brg
		normal = TBN * normalize(normal * 2.0 - 1.0);  // transform normal vector from range [0,1] to [-1,1]
	}
	
	vec3 color = texture(material.texture_diffuse1, TexCoords).rgb;
	
	// ambient
	vec3 ambient = light.ambient;
	
	// diffuse
	float diff = max(dot(lightDir, normal), 0.0);	
	vec3 diffuse = light.diffuse * diff;
	
	// specular
	vec3 halfwayDir = normalize(lightDir + viewDir);
	float spec = pow(max(dot(normal, halfwayDir), 0.0), material.shininess);
	// vec3 reflectDir = reflect(-lightDir, normal);  
	// spec = pow(max(dot(viewDir, reflectDir), 0.0), material.shininess);
	
	vec3 specular = light.specular * spec * 0.3; // reduce specularity of material
	
	
	//shadow
	float shadow = ShadowCalculation(FragPosLightSpace);
	shadow = clamp( shadow, 0.0, 1.0 );	// clamp shadow to [0 1]
	shadow = min(shadow, 0.75); 		// reduce shadow strength a little
	
	vec3 lighting = (ambient + (1.0 - shadow) * (diffuse + specular)) * color;
	fragColor = vec4(lighting.bgr,1.0f);
	
}