#version 330 core

in VS_OUT {
	vec3 fragmentPosition;
	vec3 fragmentNormal;
	vec2 fragmentUV;
	mat3 TBN;
} fs_in;

layout(location=0) out vec4 fragColor;
layout(location=1) out vec4 brightColor;

uniform vec3 diffuseColor;
uniform vec3 specularColor;
uniform int hasTexture;
uniform int hasNormalMap;
uniform int hasLighting;
uniform sampler2D color_texture;
uniform sampler2D color_normalMap;

uniform vec3 lightPosition;
uniform vec3 cameraPosition;

uniform bool celShading;

float diffuseSimple(vec3 L, vec3 N);
float specularSimple(vec3 L,vec3 N,vec3 H);

void main() 
{
	if(hasNormalMap == 1 && hasTexture == 1) {
		// obtain normal from normal map in range [0,1]
		vec3 normal = texture(color_normalMap, fs_in.fragmentUV).rgb;
		// transform normal vector to range [-1,1]
		vec3 normal_tangent_space = normalize(normal * 2.0 - 1.0);  // this normal is in tangent space
		normal_tangent_space = normalize(fs_in.TBN * normal_tangent_space);

		vec3 color = texture(color_texture, fs_in.fragmentUV).rgb;
		
		vec3 lightDir = normalize(lightPosition - fs_in.fragmentPosition);
		vec3 viewDir = normalize(cameraPosition - fs_in.fragmentPosition);
		vec3 reflectDir = reflect(-lightDir, normal_tangent_space);
		vec3 halfwayDir = normalize(lightDir + viewDir);  
	
		float ambientIntensity = 0.05f; 
		float diffuseIntensity = diffuseSimple(lightDir, normal_tangent_space); 
		float specularIntensity = specularSimple(lightDir, normal_tangent_space, halfwayDir) / 10.0f; 

		float intensity = ambientIntensity + diffuseIntensity + specularIntensity;
		float shadeIntensity; 

		// Effect: Cel Shading
		if (celShading) { 
		  shadeIntensity = ceil(intensity * 5) / 5;
		} else { 
		  shadeIntensity = intensity; 
		} 

		fragColor = vec4(color * shadeIntensity, 1.0);

	}
	else {
		vec3 color = texture(color_texture, fs_in.fragmentUV).rgb;
  
		vec3 lightDir = normalize(lightPosition - fs_in.fragmentPosition);
		vec3 viewDir = normalize(cameraPosition - fs_in.fragmentPosition);
		vec3 reflectDir = reflect(-lightDir, fs_in.fragmentNormal);
		vec3 halfwayDir = normalize(lightDir + viewDir);  
	
		float ambientIntensity = 0.05f; 
		float diffuseIntensity = diffuseSimple(lightDir, fs_in.fragmentNormal); 
		float specularIntensity = specularSimple(lightDir, fs_in.fragmentNormal, halfwayDir) / 10.0f; 

		float intensity = ambientIntensity + diffuseIntensity + specularIntensity;
		
		float shadeIntensity;

		// Effect: Cel Shading
		if (celShading) { 
		  shadeIntensity = ceil(intensity * 5) / 5; 
		} else { 
		  shadeIntensity = intensity; 
		} 
  
		fragColor = vec4(color * shadeIntensity, 1.0);
	}

	// Effect: bloom
	float brightness = dot(fragColor.rgb, vec3(0.2126, 0.7152, 0.0722)); // extract the brightness from the color (grayscale)
	if (brightness > 0.9f) {
		brightColor = vec4(fragColor.rgb, 1.0);
	} else {
		brightColor = vec4(0, 0, 0, 1);
	}
}

float diffuseSimple(vec3 L, vec3 N){
   return clamp(dot(L,N),0.0,1.0);
}

float specularSimple(vec3 L,vec3 N,vec3 H) {
   if(dot(N,L)>0) {
      return pow(clamp(dot(H,N),0.0,1.0),32.0);
   }
   return 0.0;
}



float max(float x, float y)
{
	return x > y ? x : y;
}

float dot(vec3 v1, vec3 v2)
{
	return v1.r * v2.r + v1.g * v2.g + v1.b * v2.b;
}

vec3 mul(vec3 v, float f)
{
	return vec3(v.r * f, v.g * f, v.b * f);
}

vec3 sub(vec3 v1, vec3 v2)
{
	return vec3(v1.r - v2.r, v1.g - v2.g, v1.b - v2.b); 
}

vec3 norm(vec3 v)
{
	float l = length(v);
	return vec3(v.r / l, v.g / l, v.b / l);
}

float length(vec3 v)
{
	return sqrt(v.r * v.r + v.g * v.g + v.b * v.b);
}
	
