#version 330 core

struct Material {
    vec3 ambient;
    vec3 diffuse;
    vec3 specular;
    float shininess;
}; 

struct Light {
    vec3 position;
  
    vec3 ambient;
    vec3 diffuse;
    vec3 specular;
};

in vec3 fragNormal;
in vec3 fragPos;
in vec2 Texture_uv;
in vec4 FragPosLightSpace;
in mat3 TBN;
in mat3 TBNI;

in vec3 TangentLightPos;
in vec3 TangentViewPos;
in vec3 TangentFragPos;

layout(location=0)
out vec4 fragColor;

uniform float height_scale = 0.03;

uniform vec3 viewPos;
uniform vec3 lightPos;
uniform Material material;
uniform Light light;  
uniform sampler2D texture_diffuse1;
uniform sampler2D texture_specular1;
uniform sampler2D texture_normal1;
uniform sampler2D texture_height1;

uniform int use_normal_mapping;
uniform int use_parallax_mapping;

uniform sampler2D shadowMap;

float ShadowCalculation(vec4 fragPosLS)
{
	vec3 projCoords = fragPosLS.xyz / fragPosLS.w;
    projCoords = projCoords * 0.5 + 0.5;
    float closestDepth = texture(shadowMap, projCoords.xy).r; 
    float currentDepth = projCoords.z;

	//float bias = 0.005;
	vec3 normal = normalize(fragNormal);
	vec3 lightDir = normalize(-light.position);
	float bias = max(0.05 * (1.0 - dot(normal, lightDir)), 0.005);  
	// PCF
	float shadow = 0.0;
	vec2 texelSize = 1.0 / textureSize(shadowMap, 0);
	for(int x = -1; x <= 1; ++x)
	{
		for(int y = -1; y <= 1; ++y)
		{
			float pcfDepth = texture(shadowMap, projCoords.xy + vec2(x, y) / 700.0).r; 
			shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;        
		}    
	}
	shadow /= 9.0;

	//float shadow = currentDepth - bias > closestDepth  ? 1.0 : 0.0;  

	if (projCoords.z > 1.0)
		shadow = 0.0;

    return shadow;
}

vec2 ParallaxMapping(vec2 Texture_uv, vec3 viewDir)
{
	const float minLayers = 8;
	const float maxLayers = 64;
	float numLayers = mix(maxLayers, minLayers, abs(dot(vec3(0.0, 0.0, 1.0), viewDir)));
	float layerDepth = 1.0 / numLayers;
	float currentLayerDepth = 0.0;
	vec2 P =viewDir.xy * height_scale;
	vec2 deltaTexCoords = P / numLayers;

	vec2 currentTexCoords = Texture_uv;
	float currentDepthMapValue = texture(texture_height1, currentTexCoords).r;

	while(currentLayerDepth < currentDepthMapValue)
	{
		currentTexCoords -= deltaTexCoords;
		currentDepthMapValue = texture(texture_height1, currentTexCoords).r;
		currentLayerDepth += layerDepth;
	}
	vec2 prevTexCoords = currentTexCoords + deltaTexCoords;
	float afterDepth = currentDepthMapValue - currentLayerDepth;
	float beforeDepth = texture(texture_height1, prevTexCoords).r - currentLayerDepth + layerDepth;
	
	float weight = afterDepth / (afterDepth - beforeDepth);
	vec2 finalTexCoords = prevTexCoords * weight + currentTexCoords * (1.0 - weight);

	return finalTexCoords;
}

void main() {
	
	vec3 LightPos = lightPos;
	vec3 FragPos = fragPos;

	vec3 lightDir = normalize(LightPos - FragPos);
	vec3 norm = normalize(fragNormal);
	vec3 viewDir = normalize(viewPos - FragPos);

	vec2 texture_uv = Texture_uv;

	if (use_parallax_mapping == 1) {
		viewDir = normalize(TangentViewPos - TangentFragPos);
		texture_uv = ParallaxMapping(Texture_uv, viewDir);
		//if(texture_uv.x > 1.0 || texture_uv.y > 1.0 || texture_uv.x < 0.0 || texture_uv.y < 0.0)
			//discard;
		viewDir = normalize(viewPos - FragPos);
	}

    if (use_normal_mapping == 1) {
        norm = texture(texture_normal1, texture_uv).xyz;
        norm = normalize(norm * 2.0 -1.0);
        //norm = normalize(TBN * norm);
		viewDir =TBNI * normalize(viewPos - FragPos);
		lightDir =TBNI * normalize(LightPos - FragPos);
    }

	//vec3 ambient = material.ambient * light.ambient;
	vec3 ambient = texture(texture_diffuse1, texture_uv).xyz * light.ambient;

	float diff = max(dot(norm, lightDir), 0.0);
	//vec3 diffuse = (diff * material.diffuse) * light.diffuse;
	vec3 diffuse = (diff * texture(texture_diffuse1, texture_uv).xyz) * light.diffuse;

	vec3 reflectDir = reflect(-lightDir, norm);
	float spec = pow(max(dot(viewDir, reflectDir), 0.0), material.shininess);
	vec3 specular = (material.specular * spec) * light.specular;

	float shadow = ShadowCalculation(FragPosLightSpace);
	vec3 result = ambient + (1.0 - shadow) * (diffuse + specular);

	fragColor = vec4(result, 1.0f);
}
