#pragma once

#include "volumeshop.h"

#include <QWidget>
#include <QToolButton>
#include <QFrame>
#include <QLabel>
#include <QPainter>
#include <QPaintEvent>
#include <QEvent>
#include <QBoxLayout>
#include <QResizeEvent>
#include <QVBoxLayout>
#include <QMouseEvent>
#include <QPalette>

#include "down.xpm"
#include "up.xpm"

class QSidebarHeader : public QWidget
{
	Q_OBJECT

public:

	QSidebarHeader(const QString &strLabel, QWidget *pParent = NULL, Qt::WFlags wf = 0) : QWidget(pParent,wf), m_strLabel(strLabel), m_bHighlighted(false), m_bExpanded(false)
	{
		QFontMetrics fm(font());
		setFixedHeight(int(float(fm.height()) * 1.67f + 0.5f));
	};

	virtual ~QSidebarHeader()
	{
	};

	const bool IsExpanded() const
	{
		return m_bExpanded;
	};

	void toggle()
	{
		m_bExpanded = !m_bExpanded;
		repaint();
		emit toggled(m_bExpanded);
	};

signals:

	void toggled(const bool bExpanded);

protected:

	virtual void mouseReleaseEvent ( QMouseEvent * e )
	{
		QWidget::mouseReleaseEvent(e);

		if (e->button() == Qt::LeftButton)
			toggle();	
	};

	virtual void enterEvent ( QEvent *e ) 
	{
		QWidget::enterEvent(e);
		m_bHighlighted = true;
		repaint();
	};

	virtual void leaveEvent ( QEvent *e )
	{
		QWidget::leaveEvent(e);
		m_bHighlighted = false;
		repaint();
	};

	virtual void paintEvent ( QPaintEvent *e )
	{
		QWidget::paintEvent(e);

		const int iLeftMargin = 12;
		const int iRightMargin = 4;
		const int iRound = 2;
/*
		QPixmap image(width(),height());
		image.fill(this,0,0);

		QPainter p(&image);
*/

		QPainter p(this);

		const QColor colStart(palette().light().color());
		const QColor colEnd(palette().button().color());

		const float fR = float(colEnd.red() - colStart.red()) / float(width());
		const float fG = float(colEnd.green() - colStart.green()) / float(width());
		const float fB = float(colEnd.blue() - colStart.blue()) / float(width());

		for(int i=0; i < iRound; i++)
		{
			const QColor colCurrent(int(float(colStart.red()) + fR * float(i)),int(float(colStart.green()) + fG * float(i)),int(float(colStart.blue()) + fB * float(i)));
			p.setPen(colCurrent);
			p.drawLine(i, iRound-i, i, height()-1);
		}

		for(int i=iRound; i < width()-1-iRound; i++)
		{
			const QColor colCurrent(int(float(colStart.red()) + fR * float(i)),int(float(colStart.green()) + fG * float(i)),int(float(colStart.blue()) + fB * float(i)));
			p.setPen(colCurrent);
			p.drawLine(i, 0, i, height()-1);
		}

		for(int i=width()-1-iRound; i < width(); i++)
		{
			const QColor colCurrent(int(float(colStart.red()) + fR * float(i)),int(float(colStart.green()) + fG * float(i)),int(float(colStart.blue()) + fB * float(i)));
			p.setPen(colCurrent);
			p.drawLine(i, iRound-(width()-1-i), i, height()-1);
		}

		QFont font(font());
		font.setBold(true);
		p.setFont(font);

		QRect rectangle;
		rectangle.setCoords(rect().left()+iLeftMargin,rect().top(),rect().right()-iRightMargin,rect().bottom());

		if (m_bHighlighted)
			p.setPen(palette().buttonText().color().light());
		else
			p.setPen(palette().buttonText().color());

		p.drawText(rectangle,Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine,m_strLabel);

		
		QPixmap pixImage;

		if (m_bExpanded)
		{
			static const QPixmap pixImage(up_xpm);
			p.drawPixmap(rectangle.right()-pixImage.width(),(rectangle.height()-pixImage.height()) / 2,pixImage);
		}
		else
		{
			static const QPixmap pixImage(down_xpm);
			p.drawPixmap(rectangle.right()-pixImage.width(),(rectangle.height()-pixImage.height()) / 2,pixImage);
		}
		
		p.end();
		//bitBlt(this, 0, 0, &image);
	};

	virtual void windowActivationChange ( bool oldActive )
	{
		repaint();
	};

	virtual void fontChange ( const QFont & oldFont )
	{
		QFontMetrics fm(font());
		setFixedHeight(int(float(fm.height()) * 1.67f + 0.5f));
	};

private:

	QString m_strLabel;
	bool m_bHighlighted;
	bool m_bExpanded;
};

class QSidebarContent : public QWidget
{
	Q_OBJECT

public:

	QSidebarContent (QWidget *pWidget, QWidget *pParent = NULL, Qt::WFlags wf = 0) : QWidget(pParent,wf), m_pWidget(pWidget)
	{
		QGridLayout *pLayout = new QGridLayout(this);
		m_pWidget->setParent(this);
		pLayout->addWidget(pWidget);
		m_pWidget->show();
	};

	virtual ~QSidebarContent()
	{
	};

	QWidget * GetWidget()
	{
		return m_pWidget;
	};

private:

	QWidget *m_pWidget;
};


class QSidebarItem : public QFrame
{
	Q_OBJECT

public:

	QSidebarItem(const QString &strLabel, QWidget *pWidget, QWidget *pParent = NULL, Qt::WFlags wf = 0) : QFrame(pParent,wf)
	{
		QVBoxLayout *pLayout = new QVBoxLayout(this);
		m_pHeader = new QSidebarHeader(strLabel,this);
		m_pContent = new QSidebarContent(pWidget,this);

		pLayout->setMargin(0);
		pLayout->setSpacing(0);

		((QBoxLayout*)(pParent->layout()))->insertWidget(-1,this);
		connect(m_pHeader,SIGNAL(toggled(const bool)),this,SLOT(OnHeaderToggled(const bool)));
		OnHeaderToggled(m_pHeader->IsExpanded());		

		pLayout->addWidget(m_pHeader);
		pLayout->addWidget(m_pContent);
	};

	virtual ~QSidebarItem()
	{
	};

	QSidebarHeader * GetHeader()
	{
		return m_pHeader;
	};

	QSidebarContent * GetContent()
	{
		return m_pContent;
	};

protected slots:

	void OnHeaderToggled(const bool bExpanded)
	{

		if (bExpanded)
		{
			m_pContent->show();
		}
		else
		{
			m_pContent->hide();
		}
	};

private:

	QSidebarHeader *m_pHeader;
	QSidebarContent *m_pContent;
};

class QSidebar : public QFrame
{
Q_OBJECT	

public:

	QSidebar(QWidget *pParent = NULL) : QFrame(pParent)
	{
		m_pLayout = new QVBoxLayout(this);
		m_pSpacer = new QSpacerItem(1,1,QSizePolicy::Ignored,QSizePolicy::MinimumExpanding);
		m_pLayout->addItem(m_pSpacer);
		m_pLayout->setMargin(12);
		m_pLayout->setSpacing(12);

		QPalette palette(QColor(101,121,215),QColor(101,121,215));
		setPalette(palette);
		
	};

	virtual ~QSidebar()
	{
	};


	void addItem (QWidget *pWidget, const QString &strLabel)
	{
		m_pLayout->removeItem(m_pSpacer);
		QSidebarItem *pItem = new QSidebarItem(strLabel,pWidget,this);
		
		QPalette paletteHeader(QColor(199,212,247),QColor(101,121,215));
		paletteHeader.setColor(QPalette::ButtonText,QColor(33,93,198));
		pItem->GetHeader()->setPalette(paletteHeader);

		QPalette paletteContent(QColor(214,223,247),QColor(214,223,247));
		pItem->GetContent()->setPalette(paletteContent);
		m_pLayout->addItem(m_pSpacer);
	};

private:

	QVBoxLayout *m_pLayout;
	QSpacerItem *m_pSpacer;
};
