/** \file VertexBones.h
 * \brief The header file for the 'VertexBones' class.
 */

#pragma once
// OSkA Includes
#include "OSkA.h"
// Other Includes
#include <vector>

namespace OSkA
{
	/** \brief A VertexBones is a storage for the Bone weights of a vertex.
	 *
	 * The VertexBones class is used to store the Bone weights of
	 * a single vertex. It provides two special method for gaining
	 * the top four bone weights and the corresponding indices.
	 *
	 * The vertex itself is not stored in this class. The Mesh class
	 *(which contains a list of this Bones)stores the Bone weights
	 * in a list, by the index of the corresponding vertex. So the Mesh
	 * knows the vertex, and for all plausible situations, thats the
	 * only one who needs to know it.
	 */
	class VertexBones
	{
	public:
		/** \brief Default Constructor
		 */
		VertexBones();
		/** \brief Destructor
		 */
		~VertexBones();

		/** \brief Add a new Bone weight.
		 *
		 * Adds a new Bone weight. If there is already a weight for
		 * the given Bone stored, it will be overwritten.
		 * \param bone		The id of the Bone.
		 * \param weight	The weight of the Bone for this Vertex.
		 */
		void addBone(int bone, float weight);
		/** \brief Returns the number of Bones.
		 *
		 * \return The number of Bones.
		 */
		int getBonesCount();

		/** \brief Returns the top four bone weights of this object.
		 *
		 * This method returns an float[4] array with the top four bone weights
		 * of this object. Yourself have to take care of deleting this array.
		 * \return The top four bone weights.
		 */
		float* getMaxWeights();
		/** \brief Returns the corresponding indices for the bone weights of getMaxWeights()
		 *
		 * This method returns an float[4] array with the corresponding indices for the
		 * top four bone weights of getMaxWeights(). Yourself have to take care of deleting
		 * this array. 
		 *\return The indices for the top four bone weights.
		 */
		float* getMaxWeightsIndices();
	private:
		/** \brief The list of bone weights.
		 */
		std::vector<float> boneWeights;
	};
}