/** \file Track.h
 * \brief The header file for the 'Track' class.
 */

#pragma once
// OSkA Includes
#include "Keyframe.h"
#include "Matrix.h"
// Other Includes
#include <vector>

namespace OSkA
{
	/** \brief A Track stores a list of Keyframes.
	 *
	 * The Track class is used as a simple storage for a list of Keyframe object.
	 *
	 * It provides simple methods to add new Keyframes and to access them. It also
	 * has a interpolation method which will give us the interpolated transformation
	 * matrix of the keyframes at a given point of time.
	 *
	 * The Keyframes are stored in a simple list, and will be automatically sorted
	 * by our addKeyframe-method.
	 */
	class Track
	{
	public:
		/** \brief Default Constructor.
		 */
		Track();
		/** \brief Destructor.
		 */
		~Track();

		/** \brief Adds a new Keyframe.
		 *
		 * A new Keyframe is inserted to the track, ascending sorted by the point of time.
		 * \param time		The point of time.
		 * \param transform	The transformation matrix.
		 */
		void addKeyframe(float time, Matrix* transform);
		/** \brief Returns the number of Keyframes.
		 *
		 * Returns the number of Keyframes, stored in this Track.
		 * \return The number of Keyframes.
		 */
		int getKeyframesCount();
		/** \brief Returns a Keyframe.
		 *
		 * Returns the Keyframe at the given index. All Keyframes are ascending sorted.
		 * The pointer refers to an intern Keyframe object of this class. So by changing
		 * the object you will affect this class. You don't have to delete it, this class
		 * itself will take care of it.
		 * \param index	The index of the Keyframe.
		 * \return The Keyframe at the given index.
		 */
		Keyframe* getKeyframe(int index);

		/** \brief Returns the interpolated transformation Matrix.
		 *
		 * Returns the linear interpolated transformation Matrix at at a given point of time.
		 * If there are no Keyframes, the Matrix will be an identity Matrix. If the point
		 * of time lies before the begin of the animation oder after it's ending the Matrix
		 * will be the transformation Matrix of the first or the last Keyframe.
		 * Yourself will have to delete the Matrix object refered by the returned pointer.
		 * \param time	The point of time the transformation matrix will be interpolated for.
		 * \return The interpolated transformation Matrix at the given point of time.
		 */
		Matrix* getInterpolTransform(float time);
	private:
		/** \brief A vector, containing all Keyframes of this Track.
		 */
		std::vector<Keyframe*> keyframes;
	};
}