/** \file ShaderInterface.h
 * \brief The header file for the 'ShaderInterface' interface.
 */

namespace OSkA
{
	/** \brief A ShaderInterface defines the necessary functions a shader must have.
	 *
	 * The ShaderInterface defines several functions, that will be used by the
	 * AnimationShader to interact with the shader object.
	 */

	class ShaderInterface
	{
	public:
		/** \brief Enable the shader.
		 */
		virtual void enable()=0;
		/** \brief Disable the shader.
		 */
		virtual void disable()=0;

		/** \brief Returns the location of an attribute.
		 *
		 * This method returns the location of a shader attribute variable, defined by its name.
		 * \param name	name of the attribute
		 * \return Location of the attribute as unsigned int
		 */
		virtual unsigned int getAttributeLocation(char* name)=0;
		/** \brief Returns the location of an attribute.
		 *
		 * This method returns the location of a shader uniform variable, defined by its name.
		 * \param name	name of the uniform
		 * \return Location of the uniform as unsigned int
		 */
		virtual unsigned int getUniformLocation(char* name)=0;

		/** \brief Delivers attribute data to the shader.
		 *
		 * This method delivers vertex data (stored in an an array) of a specified attribute
		 * (defined by location) to the shader. For additional information see glVertexAttribPointer.
		 * \param location	location of the attribute
		 * \param size		number of components of the vertex data (1, 2, 3 or 4)
		 * \param type		data type of the components (GL_BYTE, GL_UNSIGNED_BYTE, GL_SHORT, GL_UNSIGNED_SHORT, GL_INT, GL_UNSIGNED_INT, GL_FLOAT, or GL_DOUBLE)
		 * \param normalized	Specifies whether fixed-point data values should be normalized (GL_TRUE) or converted directly as fixed-point values (GL_FALSE) when they are accessed.
		 * \param stride	byte offset between consecutive vertex attributes
		 * \param data		pointer to the first component in the data array
		 */
		virtual void setVertexAttribPointer(unsigned int location, int size, int type, bool normalized, int stride, void* data)=0;
		/** \brief Delivers an uniform matrix to the shader.
		 *
		 * This method delivers a matrix (float array) to the uniform variable defined by location.
		 * For additional information see glUniformMatrix4fv.
		 * \param location	location of the uniform
		 * \param size		number of components in the array
		 * \param transpose	shall the matrix data be transposed when transfered to the shader?
		 * \param data		pointer to the first component in the data array
		 */
		virtual void setUniformMatrix4fv(unsigned int location, int count, bool transpose, float* data)=0;

		/** \brief Delivers an uniform float value to the shader.
		 *
		 * This method delivers a single float value to the uniform variable defined by location.
		 * For additional information see glUniform1f.
		 * \param location	location of the uniform
		 * \param value		the float value
		 */
		virtual void setUniform1f(unsigned int location, float value) = 0;
		/** \brief Delivers an uniform int value to the shader.
		 *
		 * This method delivers a single int value to the uniform variable defined by location.
		 * For additional information see glUniform1i.
		 * \param location	location of the uniform
		 * \param value		the int value
		 */
		virtual void setUniform1i(unsigned int location, int value) = 0;
	};
}
