/** \file Keyframe.h
 * \brief The header file for the 'Keyframe' class.
 */

#pragma once
// OSkA Includes
#include "OSkA.h"
#include "Matrix.h"

namespace OSkA
{
	/** \brief A Keyframe stores the transformation matrix and a corresponding point of time.
	 *
	 * The Keyframe class is used as a simple storage of Keyframe data.
	 *
	 * Each Keyframe consists of a point of time and the corresponding transformation matrix.
	 * For both we have getters and setters.
	 *
	 * Additionaly it has to be said that each Keyframe belongs to a specific bone. The itself
	 * is not stored in the Keyframe class and can also not be directly accessed if we just
	 * know the Keyframe. All Keyframes of a Bone are stored in a Track object, and all Tracks
	 * are stored in an Animation by the index of the corresponding Bone. So we can not access
	 * the Bone by the Keyframe, but if we want to know the Keyframes of a Bone(which is obviously
	 * the only usefull case)we can access them.
	 */
	class Keyframe
	{
	public:
		/** \brief Default Constructor
		 */
		Keyframe();
		/** \brief Constructor
		 *
		 * Initializes the Keyframe with the point of time and the transformation matrix.
		 * \param	time		The point of time.
		 * \param	transform	The transformation matrix.
		 */
		Keyframe(float time, Matrix* transform);
		/** \brief Destructor
		 */
		~Keyframe();

		/** \brief Sets the point of time.
		 *
		 * \param time	The point of time.
		 */
		void setTime(float time);
		/** \brief Returns the stored point of time.
		 *
		 * \return The stored point of time. Default value is 0.
		 */
		float getTime();

		/** \brief Sets the transformation matrix.
		 *
		 * \param transform	The transformation matrix.
		 */
		void setTransform(Matrix* transform);
		/** \brief Returns the stored transformation matrix.
		 *
		 * The pointer refers to an intern Matrix object of this class. So by changing
		 * the object you will affect this class. You don't have to delete it, this class
		 * itself will take care of it.
		 * \return The transformation matrix. Default value is the identity matrix.
		 */
		Matrix* getTransform();
	private:
		/** \brief The point of time.
		 */
		float time;
		/** \brief The transformation Matrix.
		 */
		Matrix* transform;
	};
}