/** \file Bone.h
 * \brief The header file for the 'Bone' class.
 */

#pragma once
// OSkA Includes
#include "OSkA.h"
#include "Matrix.h"

namespace OSkA
{
	/** \brief A Bone stores the name and the bind pose(a transformation matrix)of a Bone.
	 *
	 * The Bone class is a simple storage for a Bone of our Skeleton.
	 * Each Bone has a name, an id of its parent, and two transformation
	 * matrices for its bind pose and inverse bind pose.
	 *
	 * Each Bone has an id, though it is not stored in this class directly.
	 * The id of a Bone is used as an index in the Skeleton class.
	 *
	 * The parent id tells us which Bone is the parent of the actual bone.
	 * Some Bones might be root, then the parent id will be -1(cause usual Bones
	 * will have a parent id >=0).
	 */
	class Bone
	{
	public:
		/** \brief Default Constructor.
		 */
		Bone();
		/** \brief Destructor.
		 */
		~Bone();

		/** \brief Sets the name of the Bone.
		 *
		 * \param name	The name of the Bone.
		 */
		void setName(char* name);
		/** \brief Sets the parent of the Bone.
		 *
		 * \param parentId	The id of the Bone's parent.
		 */
		void setParentId(int parentId);
		/** \brief Returns the Bones name.
		 *
		 * \return The name of the Bone. Default is empty string.
		 */
		char* getName();
		/** \brief Returns the Bones parent id.
		 *
		 * \return The id of the Bone's parent. Default is -1;
		 */
		int getParentId();

		/** \brief Sets the bind pose of the Bone.
		 *
		 * \param transform	The bind pose of the Bone.
		 */
		void setTransform(Matrix* transform);
		/** \brief Returns the bind pose of the Bone.
		 *
		 * The pointer refers to an intern Matrix object of this class. So by changing
		 * the object you will affect this class. You don't have to delete it, this class
		 * itself will take care of it.
		 * \return The Bone's bind pose.
		 */
		Matrix* getTransform();

		/** \brief Checks if the Bone is a root Bone.
		 *
		 * This method checks if the bone has a parent of if it is a root Bone.
		 * \return true - Bone is root(has no parent), false - Bone is no root(has parent)
		 */
		bool isRoot();
	private:
		/** \brief The name of the Bone.
		 */
		char* name;
		/** \brief The id of the Bone's parent.
		 *
		 * Has a value >0 for non-root Bones and -1 for root bones.
		 */
		int parentId;
		/** \brief The bind pose of the Bone.
		 */
		Matrix* transform;
	};
}