/** \file Animation.h
 * \brief The header file for the 'Animation' class.
 */

#pragma once
// OSkA Includes
#include "OSkA.h"
#include "Track.h"
// Other Includes
#include <vector>

namespace OSkA
{
	/** \brief A Animation stores a list of Tracks and has a name and a length.
	 *
	 * The Animation class is used as a simple storage for data of a single Animation.
	 * The class stores the Animations's name, its length and a list of Tracks(up to
	 * one for each Bone).
	 */
	class Animation
	{
	public:
		/** \brief Default Constructor
		 */
		Animation();
		/** \brief Constructor.
		 *
		 * Initializes the Animation with a name and a length.
		 * \param name	The name of the Animation.
		 * \param length	The length of the Animation.
		 */
		Animation(char* name, float length);
		/** \brief Destructor.
		 */
		~Animation();

		/** \brief Sets the name of the Animation.
		 *
		 * Sets a new name for this animation. The Animation will just store
		 * the pointer, so you shall not delete it, the Animation will take care
		 * of it.
		 * \param name	The name of the Animation.
		 */
		void consumeName(char* name);
		/** \brief Sets the length of the Animation.
		 *
		 * \param length	The length of the Animation.
		 */
		void setLength(float length);
		/** \brief Returns the name of the Animation.
		 *
		 * The pointer refers to an intern char array. So by changing the array data
		 * you will affect this class. You don't have to delete it, this class
		 * itself will take care of it.
		 * \return The name of the Animation.
		 */
		char* getName();
		/** \brief Returns the length of the Animation.
		 *
		 * \return The length of the Animation.
		 */
		float getLength();

		/** \brief Adds a new Track to the Animation.
		 *
		 * A new Track for the Bone with the given index is added. The track
		 * can later accessed by this index.
		 * \param boneIndex	The index of the Bone the Track belongs to.
		 */
		void addTrack(int boneIndex);
		/** \brief Returns the Track for the given index.
		 *
		 * The pointer refers to an intern Track object of this class. So by changing
		 * the object you will affect this class. You don't have to delete it, this class
		 * itself will take care of it.
		 * \param boneIndex	The index of the Bone the Track belongs to.
		 * \return The Track of the given Bone.
		 */
		Track* getTrack(int boneIndex);
		/** \brief Returns the number of Tracks.
		 *
		 * Returns the number of Tracks stored by this Animation.
		 * \return The number of Tracks.
		 */
		int getTracksCount();
	private:
		/** \brief The name of the Animation.
		 */
		char* name;
		/** \brief The length of the Animation.
		 */
		float length;

		/** \brief The Tracks of the Animation.
		 *
		 * All Tracks are indexed by the bone id. So for a given
		 * Bone we can directly access the corresponding Track.
		 */
		std::vector<Track*> tracks;
	};
}