#include <GL/glew.h>

#include <iostream>
#include <string>
#include <math.h>
//#include <SDL/SDL.h>

#include <TgaLoader/tgaload.h>
#include <GLMLoader/glm.h>
#include <GLSLProgram/GLSLProgram.h>
#include <VBOData/DoubleVBOData.h>
#include <ErrorManager/ErrorManager.h>
#include <PinholeCamera/PinholeCamera.h>
#include <GLQuickText/glQuickText.h>

//Planes for Level of detail
#define NEAR_TF 0.1
#define FAR_TF 25

enum RENDERMODE {
	WIREFRAME = 0, DISPLACEMENT = 1, DISP_WIRE = 2,
};

GLuint terrain_texture;
GLuint displacement_texture;

GLMmodel* m_model;

CPinholeCamera m_camera;
int m_mouseButton = 0;
int m_mouseX = -1;
int m_mouseY = -1;

int m_maxPrimitives = 1 << 19;
CGLSLProgram* m_program = NULL;
CGLSLProgram* m_wireprogram = NULL;
CGLSLProgram* m_renderprogram = NULL;

int m_renderMode = DISPLACEMENT;
DoubleVBOData* m_transformFeedback;
CFeedbackBuffer* m_wirebuffer;
CFeedbackBuffer* m_renderbuffer;

int m_primitivesCount = 0;
int maxPrimitivesCount = 200000;
float scale = 1.0;
float minTextel = 0.1;


void initTexture(char* terrain, char* displacement) {

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(1, &terrain_texture);
	glGenTextures(1, &displacement_texture);

	image_t temp_image, temp_image2;
	glBindTexture(GL_TEXTURE_2D, terrain_texture);
	tgaLoad(terrain, &temp_image, TGA_FREE | TGA_DEFAULT);

	glBindTexture(GL_TEXTURE_2D, displacement_texture);
	tgaLoad(displacement, &temp_image2, TGA_FREE | TGA_DEFAULT);
	minTextel = 1.0 / temp_image2.info.width;
}

void restoreMesh() {

	m_transformFeedback->Install();
	m_program->Use();
	m_transformFeedback->BeginTransformFeedback();

	glmDraw(m_model, GLM_TEXTURE | GLM_SMOOTH);

	m_transformFeedback->EndTransformFeedback();
	m_program->Stop();

}

void initWireBuffer(void) {

	delete m_wireprogram;
	m_wireprogram = new CGLSLProgram(CGLSLProgram::VS_GS);

	m_wireprogram->InitVertexShader("Shaders/VS_render.glsl");
	m_wireprogram->InitGeometryShader("Shaders/GS_wireframe.glsl",
			GL_TRIANGLES, GL_TRIANGLE_STRIP, 16);

	m_wireprogram->LinkValidate();

	GLstateTest();
	m_wirebuffer = new CFeedbackBuffer(m_wireprogram->GetProgram(), m_maxPrimitives
			* 3 * sizeof(GLfloat) * 10, GL_TRIANGLES);
	m_wirebuffer->AddGLSLVarying("tf_data");
	m_wirebuffer->AddGLSLVarying("tf_normal");
	m_wirebuffer->AddGLSLVarying("tf_position");
	m_wirebuffer->Install();
	m_wireprogram->Use();
	m_wireprogram->SetUniform1f("scale", scale);
	m_wireprogram->Stop();
}

void initRenderBuffer(void) {

	delete m_renderprogram;
	m_renderprogram = new CGLSLProgram(CGLSLProgram::VS_GS);

	m_renderprogram->InitVertexShader("Shaders/VS_render.glsl");
	m_renderprogram->InitGeometryShader("Shaders/GS_render.glsl", GL_TRIANGLES,
			GL_TRIANGLE_STRIP, 16);

	m_renderprogram->LinkValidate();

	GLstateTest();
	m_renderbuffer
			= new CFeedbackBuffer(m_renderprogram->GetProgram(), m_maxPrimitives * 3
					* sizeof(GLfloat) * 10, GL_TRIANGLES);
	m_renderbuffer->AddGLSLVarying("tf_data");
	m_renderbuffer->AddGLSLVarying("tf_normal");
	m_renderbuffer->AddGLSLVarying("tf_position");
	m_renderbuffer->Install();
	m_renderprogram->Use();
	m_renderprogram->SetTexture("terrain_texture", terrain_texture, 0);
	m_renderprogram->SetTexture("displacement_texture", displacement_texture, 1);
	m_renderprogram->SetUniform1f("scale", scale);
	m_renderprogram->Stop();
}

void initTFeedbackBuffer(void) {

	delete m_program;
	m_program = new CGLSLProgram(CGLSLProgram::VS_GS);

	m_program->InitVertexShader("Shaders/VS_save.glsl");
	m_program->InitGeometryShader("Shaders/GS_save.glsl", GL_TRIANGLES,
			GL_TRIANGLE_STRIP, 16);

	m_program->LinkValidate();

	GLstateTest();
	m_transformFeedback = new DoubleVBOData(m_program->GetProgram(), m_maxPrimitives
			* 3 * sizeof(GLfloat) * 10, GL_TRIANGLES);
	m_transformFeedback->AddGLSLVarying("tf_data");
	m_transformFeedback->AddGLSLVarying("tf_normal");
	m_transformFeedback->AddGLSLVarying("tf_position");
	m_transformFeedback->Install();
	m_program->Use();
	m_program->SetUniform1f("maxlength", minTextel);
	m_program->SetUniform1f("nearCameraDistance", NEAR_TF);
	m_program->SetUniform1f("farCameraDistance", FAR_TF);
	m_program->Stop();

	restoreMesh();
}

void applyDisplacement() {
	m_renderbuffer->Install();
	m_renderprogram->Use();
	m_renderbuffer->BeginTransformFeedback();

	m_transformFeedback->Draw();

	m_renderbuffer->EndTransformFeedback();
	m_renderprogram->Stop();
}

void wire() {
	m_wirebuffer->Install();
	m_wireprogram->Use();
	m_wirebuffer->BeginTransformFeedback();

	m_transformFeedback->Draw();

	m_wirebuffer->EndTransformFeedback();
	m_wireprogram->Stop();
}

void feedback() {
	m_transformFeedback->swapBuffers();
	m_transformFeedback->Install();
	m_program->Use();
	m_transformFeedback->BeginTransformFeedback();

	m_transformFeedback->swapBuffers();
	m_transformFeedback->Draw();
	m_transformFeedback->swapBuffers();

	m_transformFeedback->EndTransformFeedback();
	m_program->Stop();
}

void perform_feedback() {
	restoreMesh();
	while (true) {
		int currentPrimitives = m_transformFeedback->GetPrimitiveCount();
		if (m_primitivesCount == currentPrimitives || currentPrimitives
				> maxPrimitivesCount) {
			break;
		} else {
			m_primitivesCount = currentPrimitives;
			feedback();
		}
	}
}

void setCameraPosition() {
	const float *currentCameraPosition = m_camera.GetCenterOfProjection();

	m_transformFeedback->Install();
	m_program->Use();
	m_program->SetUniform3f("cameraPosition", currentCameraPosition[0],
			currentCameraPosition[1], currentCameraPosition[2]);
	m_program->Stop();
}

void display(void) {

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	m_camera.ApplyTransform();

	setCameraPosition();

	perform_feedback();

	switch (m_renderMode) {

	case WIREFRAME:
		wire();
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		m_wirebuffer->Draw();
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		break;

	case DISPLACEMENT:
		applyDisplacement();
		m_renderbuffer->Draw();
		break;

	case DISP_WIRE:
		applyDisplacement();
		glEnable(GL_POLYGON_OFFSET_FILL);
		m_renderbuffer->Draw();
		glDisable(GL_POLYGON_OFFSET_FILL);

		wire();
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		m_wirebuffer->Draw();
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		break;

	default:
		break;

	}

	glutSwapBuffers();

}

void destroy(void) {
	int windowName;

	windowName = glutGetWindow();

	if (windowName)
		glutDestroyWindow(windowName);

	if (!CErrorManager::Instance().IsOk())
		CErrorManager::Instance().FlushErrors();

	exit(EXIT_SUCCESS);
}

void keyboard(unsigned char key, int x, int y) {
	switch (key) {

	case '1':
		m_renderMode = DISPLACEMENT;
		break;

	case '2':
		m_renderMode = WIREFRAME;
		break;

	case '3':
		m_renderMode = DISP_WIRE;
		break;

	case 27:
		destroy();

		exit(EXIT_SUCCESS);
		break;
	}

	glutPostRedisplay();
}

void mouse(int button, int state, int x, int y) {
	if (state == GLUT_DOWN)
		m_mouseButton = button;
	else
		m_mouseButton = 0;

	m_mouseX = x;
	m_mouseY = y;
}

void mouseMove(int x, int y) {
	switch (m_mouseButton) {
	case GLUT_LEFT_BUTTON:
		m_camera.Pitch(0.1f * (y - m_mouseY));
		m_camera.Yaw(0.1f * (x - m_mouseX));
		break;

	case GLUT_MIDDLE_BUTTON:
		m_camera.MoveSide(0.025 * (x - m_mouseX));
		m_camera.MoveUp(0.025f * (m_mouseY - y));
		break;

	case GLUT_RIGHT_BUTTON:
		m_camera.MoveFront(0.05f * (y - m_mouseY));
		break;
	}

	m_mouseX = x;
	m_mouseY = y;

	glutPostRedisplay();
}

void reshape(GLsizei w, GLsizei h) {
	if (h == 0)
		h = 1;

	glViewport(0, 0, w, h);
	m_camera.SetViewport(0, 0, w, h);
}

void init(int argc, char** argv) {
	glutInit(&argc, argv);
	glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);
	glutInitWindowSize(512, 512);

	glutCreateWindow("Displacement Mapping");

	glewInit();

	glClearColor(1.0f, 1.0f, 1.0f, 0.0f);
	glPolygonOffset(1.0, 1.0);

	m_camera.Create(45.0f, 0.1f, 250.0f, glutGet(GLUT_WINDOW_WIDTH), glutGet(
			GLUT_WINDOW_HEIGHT));

	m_camera.MoveFront(-5.0f);

	m_camera.ApplyTransform();

	glutDisplayFunc(display);
	glutKeyboardFunc(keyboard);
	glutMouseFunc(mouse);
	glutMotionFunc(mouseMove);
	glutReshapeFunc(reshape);
	glutIdleFunc(display);

	glEnable(GL_DEPTH_TEST);
	glLineWidth(1.7);

	initTexture(argv[2], argv[3]);

	m_model = glmReadOBJ(argv[1]);

	initRenderBuffer();
	initWireBuffer();
	initTFeedbackBuffer();
}


#include <conio.h>

int main(int argc, char **argv)
{
	if (argc < 4)
	{
		fprintf(stderr, "Not enough command-line arguments provided. Please specify a .obj, and two .tga files.\nAll files must reside in the working directory!\n");
		_getch();
		exit(1);
	}

	if (argc > 4)
		maxPrimitivesCount = atoi(argv[4]);

	if (argc > 5)
		scale = atof(argv[5]);

	init(argc, argv);

	glutMainLoop();

	delete m_program;
	delete m_wireprogram;
	delete m_renderprogram;

	return EXIT_SUCCESS;
}
