/** \file XMLTools.h
 * \brief The header file for the 'XMLTools' functions.
 */

#include <xercesc/parsers/XercesDOMParser.hpp>

namespace OSkA
{
	/** \brief Helper functions for XML parsing.
	 *
	 * This functions will be used in combination with Xerces to parse
	 * XML files. This functions save a lot of recurring source code
	 * and make the XML parsing code shorter and easier to understand.
	 */

	/** \brief Extracts x/y/z attributs out of a node.
	 *
	 * Three attributs(named 'x', 'y' and 'z')are extracted out of
	 * the node and returned in an size 3 float array.
	 * \param node	The node containing the attributes.
	 * \return Size 3 float array with the x/y/z attribute values.
	 */
	float* getXYZAttributes(XERCES_CPP_NAMESPACE::DOMNode* node);
	/** \brief Extracts a character attribute out of a node.
	 *
	 * \param node	The node containing the attribute.
	 * \param attributeName	The name of the attribute that will be extracted.
	 * \return Char Array of the attribute value.
	 */
	char* getCharAttribute(XERCES_CPP_NAMESPACE::DOMNode* node, char* attributeName);
	/** \brief Extracts a float attribute out of a node.
	 *
	 * \param node	The node containing the attribute.
	 * \param attributeName	The name of the attribute that will be extracted.
	 * \return Float value of the attribute.
	 */
	float getFloatAttribute(XERCES_CPP_NAMESPACE::DOMNode* node, char* attributeName);
	/** \brief Extracts a integer attribute out of a node.
	 *
	 * \param node	The node containing the attribute.
	 * \param attributeName	The name of the attribute that will be extracted.
	 * \return Integer value of the attribute.
	 */
	int getIntAttribute(XERCES_CPP_NAMESPACE::DOMNode* node, char* attributeName);

	/** \brief Returns a list of subordinated nodes, contained by a forwarded node.
	 *
	 * \param node	The node containing the other nodes.
	 * \param nodeName	The name of the subordinated nodes.
	 * \return XERCES_CPP_NAMESPACE::DOMNodeList* List of subordinated nodes.
	 */
	XERCES_CPP_NAMESPACE::DOMNodeList* getNodeList(XERCES_CPP_NAMESPACE::DOMDocument* node, char* nodeName);
	/** \brief Returns a list of subordinated nodes, contained by a forwarded node.
	 *
	 * \param node	The node containing the other nodes.
	 * \param nodeName	The name of the subordinated nodes.
	 * \return XERCES_CPP_NAMESPACE::DOMNodeList* List of subordinated nodes.
	 */
	XERCES_CPP_NAMESPACE::DOMNodeList* getNodeList(XERCES_CPP_NAMESPACE::DOMElement* node, char* nodeName);
}
