/** \file BoneCooker.h
 * \brief The header file for the 'BoneCooker' class.
 */

#pragma once
// OSkA Includes
#include "OSkA.h"

namespace OSkA
{
	/** \brief A BoneCooker performs some preprocessing and is then used with the Shader.
	 *
	 * The BoneCooker class is used to pass the Bone weights of a Mesh
	 * to the Shader. Cause of performance reasons we don't use the
	 * Mesh directly(what would be possible), but use the BoneCooker.
	 * The BoneCooker performs some preprocessing, which leads us to better
	 * performance.
	 *
	 * To do the preprocessing it is necessary to create an new BoneCooker
	 * and to 'cook' a Mesh. Then the BoneCooker can be used with the Shader.
	 * An un-cooked BoneCooker is almost useless.
	 */
	class BoneCooker
	{
	public:
		/** \brief Default Constructor
		 */
		BoneCooker();
		/** \brief Destructor
		 */
		~BoneCooker();

		/** \brief Here the preprocessing happens.
		 *
		 * This method performs the preprocessing. It uses a Mesh object
		 * and uses it to create the weight array and the influence array, 
		 * so we don't have to create them ever and ever again.
		 * \param mesh	A pointer to the Mesh we use for preprocessing.
		 */
		void cook(Mesh* mesh);

		/** \brief Returns a pointer to the weight array.
		 *
		 * This method returns a pointer to the weight array.
		 * It does not copy the array data, so you won't have to delete
		 * it, the BoneCooker itself will do it when wou delete him.
		 * \return The weight array.
		 */
		float* getWeightArray();
		/** \brief Returns a pointer to the influence array
		 *
		 * This method returns a pointer to the influence array.
		 * It does not copy the array data, so you won't have to delete
		 * it, the BoneCooker itself will do it when wou delete him.
		 * \return The influence array.
		 */
		float* getInfluenceArray();

		/** \brief Sets the weight array
		 *
		 * This method can be used for manually setting the weight array.
		 * The BoneCooker will store the pointer, and won't copy the
		 * data, so don't delete the array data.
		 * \param weightArray	The new weight array.
		 */
		void consumeWeightArray(float* weightArray);
		/** \brief Sets the influence array
		 *
		 * This method can be used for manually setting the influence array.
		 * The BoneCooker will store the pointer, and won't copy the
		 * data, so don't delete the array data.
		 * \param influenceArray	The new influence array.
		 */
		void consumeInfluenceArray(float* influenceArray);
	private:
		/** \brief The array of bone weights.
		 *
		 * This array contains for each vertex four bone weights(the top
		 * four weights to be exactly).
		 */
		float* weightArray;
		/** \brief The array of bone weights indices.
		 *
		 * This array contains for each vertex four bone weights indices.
		 * Each index realates to it's counterpart in the weightArray
		 * and says which Bone the weight stands for.
		 */
		float* influenceArray;
	};
}